#import <Foundation/Foundation.h>


/**
 * Protocol for the notifications fpBase can send
 */
@protocol fpBaseDelegate
/**
 * Sent when the fiscal device is fully initialized
 */
-(void)deviceConnected;
/**
 * Sent when the fiscal device is disconnected
 */
-(void)deviceDisconnected;
@end

/**
 * Fiscal device communication packet format
 */
typedef NS_ENUM(int, PACKET_FORMAT)
{
    /**
     Printers like FMP-10, having 6 status bytes. For iOS these have communication channel of com.datecs.printer.fiscal
     */
    PACKET_FORMAT_OLD=0,
    /**
     Printers like FMP-350, having 8 status bytes. For iOS these have communication channel of com.datecs.printer.fiscal.2
     */
PACKET_FORMAT_NEW=1,
};

/**
 * Base class for accessing Datecs fiscal printers. It contains only custom command implementation and status bytes. Useful for printers not yet having a dedicated SDK.
 */
@interface fpBase : NSObject <NSStreamDelegate>

/**
 Returns static instance to the class
 */
+(id)sharedInstance;

/**
 Changes character encoding for the connected printer, the default one is 1251
 */
-(void)setEncoding:(int)newEncoding;
/**
 Tries to connect to a fiscal device
 @param cmd command number, refer to the documentation
 @param data optional command parameters
 @param error pointer to NSError object, where error information is stored in case function fails. You can pass nil if you don't want that information
 @return command response string
 */
-(BOOL)connectWithStreams:(NSInputStream *)inStream outputStream:(NSOutputStream *)outStream packetFormat:(PACKET_FORMAT)format error:(NSError **)error;
/**
 Breaks fiscal device connection
 */
-(void)disconnect;

/**
 Sends a custom command to the printer and returns the result
 @param cmd command number, refer to the documentation
 @param data optional command parameters
 @param error pointer to NSError object, where error information is stored in case function fails. You can pass nil if you don't want that information
 @return command response string
 */
-(NSString *)customCommand:(int)cmd data:(NSString *)data error:(NSError **)error;

/**
 Flag that shows when the communication stream is successfully opened
 */
@property (readonly) bool streamConnected;
/**
 Flag that shows when the fiscal device is successfully initialized
 */
@property (readonly) bool deviceConnected;
/**
 Delegate, where events will be sent
 */
@property (assign) id delegate;
/**
 Direct access to the status bytes array, the size is 6 bytes for old format printers, 8 bytes for new
 */
@property (readonly) uint8_t *mSB;

@end
