package com.datecs.lineaprodemo;

import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Set;

import com.datecs.api.barcode.Barcode;
import com.datecs.api.linea.LineaPro;
import com.datecs.api.linea.LineaPro.BatteryInfo;
import com.datecs.api.linea.LineaProException;
import com.datecs.api.linea.LineaProInformation;
import com.datecs.api.printer.Printer;
import com.datecs.api.printer.ProtocolAdapter;
import com.datecs.api.rfid.ContactlessCard;
import com.datecs.api.rfid.FeliCaCard;
import com.datecs.api.rfid.ISO14443Card;
import com.datecs.api.rfid.ISO15693Card;
import com.datecs.api.rfid.RC663;
import com.datecs.api.rfid.RC663.CardListener;
import com.datecs.api.rfid.STSRICard;
import com.datecs.lineaprodemo.connectivity.BluetoothSppConnector;
import com.datecs.lineaprodemo.util.HexUtil;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Fragment;
import android.app.FragmentManager;
import android.app.FragmentManager.OnBackStackChangedListener;
import android.app.FragmentTransaction;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.SharedPreferences.OnSharedPreferenceChangeListener;
import android.graphics.Color;
import android.hardware.usb.UsbAccessory;
import android.hardware.usb.UsbManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.ParcelFileDescriptor;
import android.os.SystemClock;
import android.preference.PreferenceManager;
import android.util.Log;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.Window;
import android.widget.Toast;

public class MainActivity extends Activity implements ActionHandler {
    
    public static final String LOG_TAG = "LineaProDemo";

    private static final int ACCESSORY_DELAY = 1000;
    
    private static final int BATTERY_UPDATE = 5000;
    
    private interface LineaProRunnable {
        public void run(LineaPro lineaPro) throws LineaProException, IOException;
            
    }   
        
    private final Runnable mUpdateBatteryThread = new Runnable() {
        @Override
        public void run() {
            Thread t = new Thread(new Runnable() {            
                @Override
                public void run() {
                    try {
                        LineaPro lineaPro = mLineaPro;
                        
                        if (lineaPro != null) {
                            BatteryInfo batteryInfo = lineaPro.getBatteryInfo();
                            
                            Log.d(LOG_TAG, "Battery level is " + batteryInfo.getCapacity());
                            
                            updateBattery(batteryInfo);
                            
                            mHandler.postDelayed(mUpdateBatteryThread, BATTERY_UPDATE);
                        }
                    } catch (Exception e) { }
                }
            });            
            t.start();
        }
    };
    
    private final Handler mHandler = new Handler();
    
    private final OnSharedPreferenceChangeListener mPreferenceListener = new OnSharedPreferenceChangeListener() {
        @Override
        public void onSharedPreferenceChanged(final SharedPreferences sharedPreferences, final String key) {            
            runTask(new LineaProRunnable() {                
                @Override
                public void run(LineaPro lineaPro) throws LineaProException, IOException {
                    if (key.equals("enable_scan_button")) {
                        lineaPro.enableScanButton(mPrefs.getBoolean(key, false));            
                    }
                    
                    if (key.equals("enable_battery_charge")) {
                        lineaPro.enableBatteryCharge(mPrefs.getBoolean(key, false));            
                    }
                    
                    if (key.equals("enable_external_speaker")) {
                        lineaPro.enableExternalSpeaker(mPrefs.getBoolean(key, false));
                    }
                    
                    if (key.equals("enable_external_speaker_button")) {
                        Log.d(LOG_TAG, "Change external speaker button mode");
                        lineaPro.enableExternalSpeakerButton(mPrefs.getBoolean(key, false));
                    }
                    
                    if (key.equals("device_timeout_period")) {
                        String s = mPrefs.getString(key, "0");
                        
                        if (s.equals("0")) {
                            lineaPro.setAutoOffTime(false, 30000);
                            lineaPro.setAutoOffTime(true, 60000);
                        }
                        
                        if (s.equals("1")) {
                            lineaPro.setAutoOffTime(false, 60000);
                            lineaPro.setAutoOffTime(true, 5400000);
                        }
                    }
                    
                    if (key.equals("barcode_scan_mode")) {
                        String s = mPrefs.getString(key, "0");            
                        lineaPro.bcSetMode(Integer.parseInt(s));
                    }
                }
            }, 0);
        }
    };  
        
    private UsbManager mUsbManager;
    private FileInputStream mInputStream;
    private FileOutputStream mOutputStream;
    private ParcelFileDescriptor mFileDescriptor;
    private UsbAccessory mAccessory;
    private LineaPro mLineaPro;
    
    private MainFragment mMainFragment;
    private SharedPreferences mPrefs;
    private Menu mActionBarMenu;
    
    private final BroadcastReceiver mUsbReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();
            
            Log.d(LOG_TAG, "onReceive " + intent.getAction());
            
            if (UsbManager.ACTION_USB_ACCESSORY_DETACHED.equals(action)) {                
                UsbAccessory accessory = (UsbAccessory)intent.getParcelableExtra(UsbManager.EXTRA_ACCESSORY);
                if (accessory != null && accessory.equals(mAccessory)) {                
                    closeAccessory();
                }
            }
        }
    };
    
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        requestWindowFeature(Window.FEATURE_INDETERMINATE_PROGRESS);
        setContentView(R.layout.activity_main);
        
        Log.d(LOG_TAG, "onCreate" + getIntent());
               
        mUsbManager = (UsbManager) getSystemService(Context.USB_SERVICE);
        
        mPrefs = PreferenceManager.getDefaultSharedPreferences(this);
        mPrefs.registerOnSharedPreferenceChangeListener(mPreferenceListener);
        
        // Add the fragment to the 'fragment_container' FrameLayout
        mMainFragment = new MainFragment();
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();        
        fragmentTransaction.add(R.id.fragment_container, mMainFragment, "main_fragment");        
        fragmentTransaction.commit();
        
        // Check for attached accessory
        final UsbAccessory[] accessories = mUsbManager.getAccessoryList();
        if (accessories != null && accessories.length > 0) {
            mHandler.postDelayed(new Runnable() {                
                @Override
                public void run() {
                    openAccessory(accessories[0]);
                }
            }, ACCESSORY_DELAY);
        }
    }

    /** Called when the activity is destroyed. */
    @Override
    public void onDestroy() {
        super.onDestroy();
        
        Log.d(LOG_TAG, "onDestroy");        
        
        mPrefs.unregisterOnSharedPreferenceChangeListener(mPreferenceListener);
        
        closeAccessory();
    }
    
    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.main, menu);
        mActionBarMenu = menu;
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // Handle item selection
        switch (item.getItemId()) {
            case R.id.menu_clear_log: {
                AlertDialog dialog = new AlertDialog.Builder(this)
                    .setTitle(R.string.app_name)
                    .setMessage(R.string.msg_question_clear_log)
                    .setPositiveButton(android.R.string.yes, new DialogInterface.OnClickListener() {
                        
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            clearLog();
                        }
                    })
                    .setNegativeButton(android.R.string.no, new DialogInterface.OnClickListener() {
                        
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            dialog.dismiss();
                        }
                    })
                    .create();
                
                    dialog.show();
                return true;
            }
            case R.id.menu_print_receipt: {
                String text = mMainFragment.getLog();
                printReceipt(text);
                return true;
            }
            default:
                return super.onOptionsItemSelected(item);
        }
    }
    
    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        
        Log.d(LOG_TAG, "onNewIntent " + intent.getAction());        
        
        String action = intent.getAction();

        if (UsbManager.ACTION_USB_ACCESSORY_ATTACHED.equals(action)) {
            mHandler.removeCallbacksAndMessages(null);
            mHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    closeAccessory();
                    UsbAccessory accessory = mUsbManager.getAccessoryList()[0];
                    openAccessory(accessory);
                }
            }, 1000);
        }
    }
    
    public void startScan() {
        runTask(new LineaProRunnable() {            
            @Override
            public void run(LineaPro lineaPro) throws LineaProException, IOException {
                lineaPro.bcStartScan();     
            }
        }, 0);
    }
    
    public void stopScan() {
        runTask(new LineaProRunnable() {            
            @Override
            public void run(LineaPro lineaPro) throws LineaProException, IOException {
                lineaPro.bcStopScan();     
            }
        }, 0);
    }
    
    public void turnOff() {
        runTask(new LineaProRunnable() {            
            @Override
            public void run(LineaPro lineaPro) throws LineaProException, IOException {
                lineaPro.turnOff();                
            }
        }, 0);  
    }
    
    public void readCard() {
        runTask(new LineaProRunnable() {            
            @Override
            public void run(LineaPro lineaPro) throws LineaProException, IOException {                
                final InputStream rfidInputStream = lineaPro.rfidGetInputStream(LineaPro.RFID_MODULE_CLRC663);
                final OutputStream rfidOutputStream = lineaPro.rfidOutputStream(LineaPro.RFID_MODULE_CLRC663);
                final RC663 rc663 = new RC663(rfidInputStream, rfidOutputStream);                               
                rc663.setCardListener(new CardListener() {                    
                    @Override
                    public void onCardDetect(ContactlessCard card) {
                        
                        logCard(card);
                        
                        synchronized (rc663) {
                            rc663.notify();
                        }
                    }
                });
                rc663.enable();
                
                synchronized (rc663) {
                    try {
                        rc663.wait(10000);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }
                
                rc663.disable();
                rc663.close();
            }
        }, R.id.panel_rfid);  
    }
    
    public void readInfo() {
        runTask(new LineaProRunnable() {            
            @Override
            public void run(LineaPro lineaPro) throws LineaProException, IOException {
                final LineaProInformation info = lineaPro.getInformation();
                logI(info.getName() + " " + info.getVersion() + "\n");
                
                if (info.hasBarcodeEngine()) {
                    logI("Barcode: ");
                    
                    if (info.hasOpticodeEngine()) {
                        logD("Opticon Barcode Engine\n");
                    }
                    if (info.hasNewlandEngine()) {
                        logD("NewLand Barcode Engine\n");
                    }
                } 
                
                if (lineaPro.isScanButtonEnabled()) {
                    logD("Scan button is enabled\n");
                }
                
                if (lineaPro.isBatteryChargeEnabled()) {
                    logD("Battery charge is enabled\n");
                }
                                
                if (info.hasExternalSpeaker()) {
                    if (lineaPro.isExternalSpeakerEnabled()) {
                        logD("External speaker is enabled\n");
                    }
                    
                    if (lineaPro.isExternalSpeakerButtonEnabled()) {
                        logD("External speaker button is enabled\n");
                    }
                }
                
                int autoOffTimeout1 = lineaPro.getAutoOffTime(false) / 1000;
                int autoOffTimeout2 = lineaPro.getAutoOffTime(true) / 1000;
                logD("Device timeout is " + autoOffTimeout1 + "s/" + autoOffTimeout2 + "s\n");
                
                int barcodeMode = lineaPro.bcGetMode();
                String[] modes = getResources().getStringArray(R.array.pref_barcode_scan_mode_titles);
                if (barcodeMode < modes.length) {
                    logD(modes[barcodeMode] + "\n");
                }
                
                logD("\n");
            }
        }, R.id.panel_progress);  
    }
    
    public void readBatteryInfo() {
        runTask(new LineaProRunnable() {            
            @Override
            public void run(LineaPro lineaPro) throws LineaProException, IOException {
                final BatteryInfo battery = lineaPro.getBatteryInfo();
                final BatteryInfo.Fuelgauge fuelgauge = battery.getFuelgauge();
                
                logI("Battery Information\n");
                if (fuelgauge != null) {
                    logD("Temperature: " + fuelgauge.getTemperature() + "K\n");
                    logD("Internal temperature: " + fuelgauge.getInternalTemperature() + "K\n");
                    logD("Voltage: " + fuelgauge.getVoltage() + "mV\n");
                    logD("Nominal available capacity: " + fuelgauge.getNominalAvailableCapacity() + "mAh\n");
                    logD("Full available capacity: " + fuelgauge.getFullAvailableCapacity() + "mAh\n");
                    logD("Remaining capacity: " + fuelgauge.getRemainingCapacity() + "mAh\n");
                    logD("Full charge capacity: " + fuelgauge.getFullChargeCapacity() + "mAh\n");
                    logD("Average current: " + fuelgauge.getAverageCurrent() + "mA\n");
                    logD("Standby current: " + fuelgauge.getStandbyCurrent() + "mA\n");
                    logD("Max load current: " + fuelgauge.getMaxLoadCurrent() + "mA\n");
                    logD("Average power: " + fuelgauge.getAveragePower() + "mW\n");
                    logD("State of charge: " + fuelgauge.getStateOfCharge() + "%\n");
                    logD("State of health: " + fuelgauge.getStateOfHealth() + "%\n");                                       
                } else {
                    logD("Voltage: " + battery.getVoltage() + "mV\n");                    
                    logD("Capacity: " + battery.getCapacity() + "%\n");
                    logD("Initial capacity: " + battery.getInitialCapacity() + "mAh\n");
                    logD("State of health: " + battery.getHealthLevel() + "%\n");
                    if (battery.isCharging()) {
                        logD("Battery is charging\n");
                    }                    
                }
                
                logD("\n");
            }
        }, R.id.panel_progress);  
    }
    
    @Override
    public void showSettings() {
        final FragmentManager manager = getFragmentManager();
        manager.addOnBackStackChangedListener(new OnBackStackChangedListener() {            
            @Override
            public void onBackStackChanged() {
                Fragment fragment = manager.findFragmentByTag("settings_fragment");
                
                Log.d(LOG_TAG, "Fragment backstack is changed and settings fragment is " + fragment);
                
                if (fragment == null) {
                    mMainFragment.getView().setVisibility(View.VISIBLE);   
                    manager.removeOnBackStackChangedListener(this);
                }
            }
        });
        
        SettingsFragment fragment = new SettingsFragment();
        FragmentTransaction transaction = manager.beginTransaction();
        transaction.setCustomAnimations(android.R.animator.fade_in, android.R.animator.fade_out);
        transaction.add(R.id.fragment_container, fragment, "settings_fragment");
        transaction.addToBackStack(null);
        transaction.commit();
        
        mMainFragment.getView().setVisibility(View.INVISIBLE);
    }    
     
    @Override
    public void clearLog() {
        Log.d(LOG_TAG, "Clear log");
        
        runOnUiThread(new Runnable() {            
            @Override
            public void run() {
                mMainFragment.clearLog();
            }
        });        
    }
    
    @Override
    public void logD(final String text) {
        Log.d(LOG_TAG, "Log debug: " + text);
        
        runOnUiThread(new Runnable() {            
            @Override
            public void run() {
                mMainFragment.addLog(text, Color.BLACK, false);
            }
        });        
    }
    
    @Override
    public void logI(final String text) {
        Log.d(LOG_TAG, "Log info: " + text);
        
        runOnUiThread(new Runnable() {            
            @Override
            public void run() {
                mMainFragment.addLog(text, Color.BLACK, true);
            }
        });              
    }
    
    @Override
    public void logE(final String text) {
        Log.d(LOG_TAG, "Log error: " + text);
        
        runOnUiThread(new Runnable() {            
            @Override
            public void run() {
                mMainFragment.addLog(text, Color.RED, true);
            }
        });           
    }
    
    @Override
    public void logW(final String text) {
        Log.d(LOG_TAG, "Log warning: " + text);
        
        runOnUiThread(new Runnable() {            
            @Override
            public void run() {
                mMainFragment.addLog(text, 0xFFFF7F00, false);
            }
        });            
    }
    
    private void logCard(ContactlessCard card) {
        logI("Card: ");
        logD(card.getTypeName() + "\n");
        logI("UID: ");
        logD(HexUtil.byteArrayToHexString(card.uid) + "\n");
        
        if (card instanceof ISO14443Card) {            
            // ISO14443Card iso14443Card = (ISO14443Card)card;
            
            
        } else if (card instanceof ISO15693Card) {
            // ISO15693Card iso15693Card = (ISO15693Card)card;
            
            logI("Block size: ");
            logD("" + card.blockSize + "\n");
            logI("Max blocks: ");
            logD("" + card.maxBlocks + "\n");
        } else if (card instanceof FeliCaCard) {
            // FeliCaCard feliCaCard = (FeliCaCard)card;
                        
        } else if (card instanceof STSRICard) {
            // STSRICard stsriCard = (STSRICard)card;
                        
            logI("Block size: ");
            logD("" + card.blockSize + "\n");                                 
        }
        
        logD("\n");                   
    }
      
    @Override
    public void resetBarcodeEngine() {
        runTask(new LineaProRunnable() {            
            @Override
            public void run(LineaPro lineaPro) throws LineaProException, IOException {
                lineaPro.bcRestoreDefaultMode();
            }
        }, R.id.panel_progress); 
    }
    
    @Override
    public void switchLeds(final int color) {
        runTask(new LineaProRunnable() {
            @Override
            public void run(LineaPro lineaPro) throws LineaProException, IOException {
                switch (color) {
                    case 0: {
                        lineaPro.setLED(false, true, false);
                        break;
                    }
                    case 1: {
                        lineaPro.setLED(true, false, false);
                        break;
                    }
                    case 2: {
                        lineaPro.setLED(true, true, false);
                        break;
                    }
                    case 3: {
                        lineaPro.setLED(false, false, true); 
                        break;
                    }
                }
                
                SystemClock.sleep(2000);
                
                lineaPro.setLED(false, false, false); 
            }
        }, R.id.panel_progress); 
    }
    
    private void showPanel(int id, boolean visible) {
        final View v = findViewById(id);
        final int visibility = visible ? View.VISIBLE : View.INVISIBLE; 
                
        runOnUiThread(new Runnable() {            
            @Override
            public void run() {
                v.setVisibility(visibility);
            }
        });        
    }
    
    private void updateBattery(final BatteryInfo batteryInfo) {
        runOnUiThread(new Runnable() {            
            @Override
            public void run() {
                mMainFragment.setBattery(batteryInfo);
            }
        });
        
    }
    
    private void logBarcode(Barcode barcode) {
        Log.d(LOG_TAG, "Barcode: " + barcode);

        logI("Barcode: ");
        logD(barcode.getTypeString() + "\n");
        logI("Data: ");
        
        // Represent barcode data in human readable format.
        char[] data = barcode.getDataString().toCharArray();
        
        for (int i = 0; i < data.length; i+=24) {
            int count = Math.min(24,  data.length - i);
            
            if (i != 0) {
                logD("      ");
            }
            logD(new String(data, i, count) + "\n");
        }
        logD("\n");
        
        if (mPrefs.getBoolean("beep_upon_scan", false)) {
            runTask(new LineaProRunnable() {                
                @Override
                public void run(LineaPro lineaPro) throws LineaProException, IOException {
                    lineaPro.beep(100, new int[] { 2730, 150, 65000, 20, 2730, 150 });
                }
            }, 0);
        }
        
        if (mPrefs.getBoolean("vibrate_on_barcode_scan", false)) {
            runTask(new LineaProRunnable() {                
                @Override
                public void run(LineaPro lineaPro) throws LineaProException, IOException {
                    lineaPro.startVibrator(100);
                }
            }, 0);
        }        
    }
    
    private void runTask(final LineaProRunnable r, final int id) {
        final Thread t = new Thread(new Runnable() {           
            @Override
            public void run() {                
                synchronized (MainActivity.this) {                    
                    if (id != 0) {
                        showPanel(id, true);
                    }                    
                    try {    
                        LineaPro lineaPro = mLineaPro;                    
                        if (lineaPro != null) {
                            r.run(lineaPro);
                        }
                    } catch (LineaProException e) {
                        logW(e.getMessage() + "\n\n");                    
                    } catch (IOException e) {
                        logE(e.getMessage() + "\n\n");                    
                    } catch (Exception e) {
                        logE(e.getMessage() + "\n\n");                    
                    } finally {
                        if (id != 0) {
                            showPanel(id, false);
                        }
                    }
                }
            }
        });
        t.start();
    }
   
    private void openAccessory(UsbAccessory accessory) {
        Log.d(LOG_TAG, "Open accessory");
        
        if (isFinishing()) return;
        
        if (mUsbManager.hasPermission(accessory)) {
            mFileDescriptor = mUsbManager.openAccessory(accessory);        
            if (mFileDescriptor != null) {
                mAccessory = accessory;
                FileDescriptor fd = mFileDescriptor.getFileDescriptor();
                mInputStream = new FileInputStream(fd);
                mOutputStream = new FileOutputStream(fd);
                mLineaPro = new LineaPro(mInputStream, mOutputStream);
                
                IntentFilter filter = new IntentFilter(UsbManager.ACTION_USB_ACCESSORY_DETACHED);            
                registerReceiver(mUsbReceiver, filter);
                
                showPanel(R.id.panel_rfid, false);
                showPanel(R.id.panel_barcode, false);
                showPanel(R.id.panel_progress, false);                        
                showPanel(R.id.panel_connection, false);
                
                clearLog();
                
                initialize();
            } else {
                Log.w(LOG_TAG, "Open accessory failed");
            }
        } else {
            Log.e(LOG_TAG, "No permition to open accessory");
        }
    }

    private void closeAccessory() {
        if (mFileDescriptor != null) {
            Log.e(LOG_TAG, "Close accessory");
            try {
                mLineaPro.turnOff();
            } catch (IOException e)  {
                e.printStackTrace();
            }
            try {
                mFileDescriptor.close();
            } catch (IOException e) {
                e.printStackTrace();
            }

            mFileDescriptor = null;

            unregisterReceiver(mUsbReceiver);
            
            showPanel(R.id.panel_rfid, false);
            showPanel(R.id.panel_barcode, false);
            showPanel(R.id.panel_progress, false);                        
            showPanel(R.id.panel_connection, true);
        }
    }
    
    private void initialize() {
        final LineaPro lineaPro = mLineaPro;
        
        final Thread t = new Thread(new Runnable() {           
            @Override
            public void run() {  
                try {                    
                    final LineaProInformation lineaProInfo = lineaPro.getInformation();
                    
                    lineaPro.setConnectionListener(new LineaPro.ConnectionListener() {                        
                        @Override
                        public void onDisconnect() {
                            showPanel(R.id.panel_connection, true);
                        }
                    });
                    
                    lineaPro.setBarcodeListener(new LineaPro.BarcodeListener() {
                        @Override
                        public void onReadBarcode(Barcode barcode) {
                            logBarcode(barcode);
                        }
                    });
                    
                    lineaPro.setButtonListener(new LineaPro.ButtonListener() {                            
                        @Override
                        public void onButtonStateChanged(int index, boolean state) {
                            if (mPrefs.getBoolean("enable_scan_button", false) && index == 0) {
                                if (state) {
                                    showPanel(R.id.panel_barcode, true);
                                } else {
                                    showPanel(R.id.panel_barcode, false);
                                }
                            }
                        }
                    });
                           
                    // Load known settings from device
                    boolean scanButtonEnabled = lineaPro.isScanButtonEnabled();
                    mPrefs.edit().putBoolean("enable_scan_button", scanButtonEnabled).commit();
                                            
                    boolean batteryChargeEnabled = lineaPro.isBatteryChargeEnabled();
                    mPrefs.edit().putBoolean("enable_battery_charge", batteryChargeEnabled).commit();
                    
                    if (lineaProInfo.hasExternalSpeaker()) {
                        boolean externalSpeakerEnabled = lineaPro.isExternalSpeakerEnabled();
                        mPrefs.edit().putBoolean("enable_external_speaker", externalSpeakerEnabled).commit();
                        
                        boolean externalSpeakerButtonEnabled = lineaPro.isExternalSpeakerButtonEnabled();
                        mPrefs.edit().putBoolean("enable_external_speaker_button", externalSpeakerButtonEnabled).commit();
                    }
                    
                    int autoOffTime = lineaPro.getAutoOffTime(false);
                    if (autoOffTime == 30000 /* 30 sec */) {
                        mPrefs.edit().putString("device_timeout_period", "0").commit();
                    } else {
                        mPrefs.edit().putString("device_timeout_period", "1").commit();
                    }
                    
                    int barcodeMode = lineaPro.bcGetMode();
                    mPrefs.edit().putString("barcode_scan_mode", "" + barcodeMode).commit();
                    
                    // Enable battery update timer
                    mHandler.postDelayed(mUpdateBatteryThread, 1000);
                                        
                    // Read information.
                    readInfo();
                    
                } catch (Exception e) {
                    e.printStackTrace();
                }               
            }
        });      
        t.start();
    }   
    
    private void printReceipt(final String text) {
        final BluetoothAdapter btAdapter = BluetoothAdapter.getDefaultAdapter();
        
        if (btAdapter == null) {
            Toast.makeText(this, R.string.msg_bluetooth_not_supported, Toast.LENGTH_LONG).show();
            return;
        }
        
        if (!btAdapter.isEnabled()) {
            Toast.makeText(this, R.string.msg_bluetooth_not_enabled, Toast.LENGTH_LONG).show();
            return;
        }
        
        Set<BluetoothDevice> btDevices = btAdapter.getBondedDevices();
        BluetoothDevice btDevice = null;
        for (BluetoothDevice device: btDevices) {
            if (device.getName().contains("DPP")) {
                btDevice = device;
                break;
            }
        }
        
        if (btDevice == null) {
            Toast.makeText(this, R.string.msg_printer_not_configured, Toast.LENGTH_LONG).show();
            return;
        }
        
        final BluetoothSppConnector btConnector = new BluetoothSppConnector(this, btAdapter, btDevice);
        
        mActionBarMenu.findItem(R.id.menu_print_receipt).setVisible(false);
        setProgressBarIndeterminateVisibility(true);
        
        final Thread t = new Thread(new Runnable() {            
            @Override
            public void run() { 
                try {
                    btConnector.connect();
                       
                    InputStream inputStream = btConnector.getInputStream();
                    OutputStream outputStream = btConnector.getOutputStream();
                    ProtocolAdapter protocolAdapter = new ProtocolAdapter(inputStream, outputStream);
                    Printer printer = null;
                    
                    if (protocolAdapter.isProtocolEnabled()) {
                        ProtocolAdapter.Channel printerChannel = protocolAdapter.getChannel(ProtocolAdapter.CHANNEL_PRINTER);
                        printer = new Printer(printerChannel.getInputStream(), printerChannel.getOutputStream());
                    } else {
                        printer = new Printer(protocolAdapter.getRawInputStream(), protocolAdapter.getRawOutputStream());
                    }
                    
                    printer.printTaggedText("{reset}{center}*** LineaPro Log Begin ***{br}{br}");
                    printer.printText(text);
                    printer.printTaggedText("{reset}{center}*** LineaPro Log End ***{br}");
                    printer.feedPaper(110);
                    
                    printer.close();
                    protocolAdapter.close();
                    
                    runOnUiThread(new Runnable() {                        
                        @Override
                        public void run() {
                            Toast.makeText(getApplicationContext(), R.string.msg_printing_successfully, Toast.LENGTH_LONG).show();
                        }
                    });
                } catch (IOException e) {
                    final String msg = getString(R.string.msg_printing_failed) + e.getMessage();
                    
                    runOnUiThread(new Runnable() {                        
                        @Override
                        public void run() {
                            Toast.makeText(getApplicationContext(), msg, Toast.LENGTH_LONG).show();
                        }
                    });
                } finally {
                    try {
                        btConnector.close();
                    } catch (IOException e) {                        
                        e.printStackTrace();
                    }
                    
                    runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            mActionBarMenu.findItem(R.id.menu_print_receipt).setVisible(true);
                            setProgressBarIndeterminateVisibility(false);
                        }
                    });
                }
            }
        });
        t.start();
    }

}
