/*
 * PrintDriverSDK.h 
 * 
 * Copyright 2007 Datecs Ltd. All rights reserved.
 *
 * This header file defines the interface that the PrintDriver library
 * presents to client programs.
 *
*/

#ifndef _PDSDK_H_
#define _PDSDK_H_

// Make sure we can call this stuff from C++.
#ifdef __cplusplus
extern "C" {
#endif

// Various type definitions.
#ifndef INT_BOOL
#define INT_BOOL INT
#endif


// The version of PrintDriver library.
#ifdef PDSDK_VERSION
# undef PDSDK_VERSION
#endif
#define PDSDK_VERSION     "3.16.0"

/*
 * The format of the version string is "X.Y.Z<trailing string>", where
 * X is the major version number, Y is the minor version number and Z
 * is the release number. The trailing string is often "alpha" or "beta".
 * For example "2.0.1beta".
 *
 * The PDSDK_VERSION_NUMBER is an integer with the value 
 * (X*100000 + Y*1000 + Z). For example, for version "2.0.1beta", 
 * PDSDK_VERSION_NUMBER is set to 2000001. To detect if they are using 
 * version 2.0.1 or greater at compile time, programs may use the test 
 * (PDSDK_VERSION_NUMBER >= 2000001).
*/
#ifdef PDSDK_VERSION_NUMBER
# undef PDSDK_VERSION_NUMBER
#endif
#define PDSDK_VERSION_NUMBER 3016000

/**
 * The version string is also compiled into the library so that a program
 * can check to make sure that the lib*.a file and the *.h file are from
 * the same version.  The printerdriver_libversion() function returns a 
 * pointer to the printdriver_version variable - useful in DLLs which cannot 
 * access global variables. 
*/
extern const char PDSDK_version[];
const char *PDSDK_LibVersion(void);

/**
 Return the value of the PDSDK_VERSION_NUMBER macro when the
 library was compiled.
*/
int PDSDK_LibVersion_Number(void);

/* Return values for PDSDK_* functions. */
#define PDSDK_ENONE			   0 /* Successful result. */
#define PDSDK_EGENERAL        -1 /* General error. */
#define PDSDK_ENOIMPLEMENTED  -2 /* Not implemented. */
#define PDSDK_EINTERRUPTED    -3 /* Interrupted. */
#define PDSDK_EBUSY           -4 /* Busy. */
#define PDSDK_EMEMORY         -5 /* Memory allocation problem. */
#define PDSDK_ETIMEOUT        -6 /* Timeout. */
#define PDSDK_EINVALIDPARAMS  -7 /* Invalid parameters. */
#define PDSDK_EFAILED		  -8 /* Command failed. */
/* End of error codes definition. */

/**
 This function cause open connection with printer.

 @note This function must be call before any other PDSDK_* functions. If previous connection 
 is opened then it shall close it.

 @param bKernelMode Indicate to use printer driver in kernel mode. In this mode any call to 
 printer is pass through graphics display interface. Usually this parameter must be FALSE 
 and is left only for compatibility with older version.
 @param lpszPrinterName Indicate which printer to use. Printer must have a valid registry 
 record in [HKLM\Printers]. If value is NULL or empty string then default printer is used.
    
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_OpenPrinter(BOOL bKernelMode, LPCTSTR lpszPrinterName);

/* Define printer device types. */
#define PDSDK_PRINTER_PP50    1 /* Datecs PP50. */
#define PDSDK_PRINTER_PP55    2 /* Datecs PP55. */
#define PDSDK_PRINTER_DPP350  3 /* Datecs DPP350. */
#define PDSDK_PRINTER_CMP10   4 /* Citizen CMP10. */
#define PDSDK_PRINTER_DPP250  5 /* Datecs DPP250. */
#define PDSDK_PRINTER_PP60	  6 /* Datecs PP60. */
#define PDSDK_PRINTER_BL112   7 /* Sanei BL-112. */
#define PDSDK_PRINTER_EP60    8 /* Datecs EP60. */
#define PDSDK_PRINTER_DPP450  9 /* Datecs DPP450. */

/* Define printer communication modes. 
 * 
 * Datecs PP50 can works only in raw mode.
 * Datecs PP55 can works in raw or protocol mode.
 * Datecs DPP350 can works in raw or protocol modedepends of hardware switch 4.
 * Datecs DPP250 can works in raw or protocol modedepends of hardware switch 4.
 * Citizen CMP10 can works only in raw mode. 
 * Sanei BL-112 can works only in raw mode. 
 * Datecs EP60 can works only in raw mode. 
*/
#define PDSDK_MODE_RAW        1 /* Printer in raw mode. */
#define PDSDK_MODE_PROTOCOL   2 /* Printer in protocol mode. */
/* End of printer modes definitions. */

/* Define intensity levels of printed output. */
#define PDSDK_INTENSITY_70    0 /* 70%. */
#define PDSDK_INTENSITY_80    1 /* 80%. */
#define PDSDK_INTENSITY_90    2 /* 90%. */
#define PDSDK_INTENSITY_100   3 /* 100%. */
#define PDSDK_INTENSITY_120   4 /* 120%. */
#define PDSDK_INTENSITY_150   5 /* 150%. */
/* End of intensity levels definitions. */

/* Define printed logo position. */
#define PDSDK_PRINTLOGO_NONE    0 /* No print logo. */
#define PDSDK_PRINTLOGO_TOP     1 /* Print logo on start document. */
#define PDSDK_PRINTLOGO_BOTTOM  2 /* Print logo on end document. */
/* End of printed logo position definitions. */

/* Printer settings structure. */
#pragma pack(1)
typedef struct {		
	int nSize;          /* Size of this structure. */
	int nPrinterType;   /* Type of printer device. */
	int nPrinterMode;   /* Printer mode. */	

	/*
	 * Next settings are usable when print graphics via 
	 * PDSDK_StartPage and PDSDK_EndPage.	 
    */		
	INT_BOOL bDithering; /* Dithering applied over image. */		
	int nBrightness;     /* Brightness value in percent from 0 to 100%. */	
	int nContrast;       /* Contrast value in percent from 0 to 100%. */
	int nIntensity;		 /* Intensity level. */
	int nPrintLogoPos;	 /* Print logo position. */		
	INT_BOOL bPrintWholePage; /* Print whole graphics page with white space 
							   * at the end of printer page size. */	
	int nFeedAfterPrint; /* Feed paper with n lines after end document. */

	/* Next settings are usable everytime. */    	
	INT_BOOL bShowProgress; /* Show progress cursor while works with 
							 * printer. */ 	
	int nLogLevel;          /* Log level from 0 to 5. Higher is more verbose.
	                         * The log content is stored into "\\PrintDriver.txt".
							 */ 	
} 
PDSDK_PRINTER_SETTINGS;
#pragma pack()
/* End of type definition. */

/**
 This function cause open connection with printer.
 
 @note This function must be call before any other PDSDK functions. If previous connection is opened then 
 it shall close it.
 @param lpSettings Long pointer to structure that contains printer driver settings. 
 @param lpszLogAddress Long pointer to buffer that contains logical address used for connection.
    Some of the valid values are:
      "COM1:"
      "COM1: 115200" 
      "COM9:"
      "BTH:"
      "IRDA:"
      "RNDIS:" 
 @param lpszExtraData Specifies extra data that function expect depends of "lpszLogAddress" 
 parameter. For example if you use "BTH:" logical connection type you must supply bluetooth address 
 of the remote device in TCHAR string format: "XX:XX:XX:XX:XX:XX", where XX if value from 0x00 to 0xFF.  

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.   
*/
int PDSDK_OpenPrinter2(PDSDK_PRINTER_SETTINGS *lpSettings, LPCTSTR lpszLogAddress, LPCTSTR lpszExtraData);

/**
 This function cause close connection with printer.    
 
 @note Must be call to free resources that PDSDK_OpenPrinter or PDSDK_OpenPrinter2 allocates.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.    
*/
int PDSDK_ClosePrinter(void);

/**
 This function check for active connection with printer.  
     
 @return If you have active connection opened with printer this function returns non zero value. 
 In other case function returns zero. 
*/
INT_BOOL PDSDK_IsActive(void);

/**
  This function cause stop and immediate exit from functions. 
    
  @note You can use that function for immediate exit from PDSDK_ReadMagstripe, PDSDK_ReadBarcode and etc.
*/
void PDSDK_ForceStop(void);

/* Bluetooth device information data. */
#pragma pack(1)
typedef struct 
{
	TCHAR szAddr[32]; /* Device address in format XX:XX:XX:XX:XX:XX. */
	TCHAR szName[32]; /* Device name. */
} 
PDSDK_BTH_INFO_DATA;
#pragma pack()
/* End of type definition. */

/**
 This set of functions enumerate bluetooth device in range.
 
 PDSDK_FindFirstBthDevice must be call first to start searching. PDSDK_FindNextBthDevice 
 continues searching. To enumerate all devices you need to call it until result is greater
 then zero. PDSDK_FindClose close searching and free allocated data.    

 @param pBthInfo Long pointer to a variable that receives bluetooth device information data.  
 
 @return The return value can be equal to 0 if there are no more devices, 
 greater then 0 if bluetooth device found, or less then 0 if there 
 is an error. The particular return value depends on the type of error.  
*/
int PDSDK_FindFirstBthDevice(PDSDK_BTH_INFO_DATA *pBthInfo);
int PDSDK_FindNextBthDevice(PDSDK_BTH_INFO_DATA *pBthInfo);
int PDSDK_FindClose(void);

/**
 This function retrieves a handle to a printer device context.  
 
 @note You can use this device context with GDI functions to draw into printer device bitmap.

 @return The return value is a handle to the device context if there are no errors and NULL 
 if there is an error.  
*/
HDC PDSDK_GetDC(void);

/**
 This function returns pointer to printer port object.
 
 @note Supplied for internal use only.

 @return pointer to printer port object. 
*/
const void *PDSDK_GetPrinterPortReference(void);

/**
 This function return pointer to printer settings.
 
 @note Supplied for internal use only.

 @return pointer to printer settings.  
*/
const void *PDSDK_GetPrinterSettingsReference(void);

/**
 This function cause write data direct to printer port.
 
 If printer is into protocol mode you must send and protocol header data.

 @param lpBuffer Long pointer to a buffer that contains data to be write. 
 @param nNumberOfBytesToWrite Specifies length in bytes of data to be write.   
 
 @return the length of bytes that were successfully write to printer port if there are no 
 errors and negative value if there is an error. The particular return value depends 
 on the type of error. 
*/
int PDSDK_WritePortDirect(LPCVOID lpBuffer, int nNumberOfBytesToWrite);

/**
 This function cause read data direct from printer port.
 
 If printer is into protocol mode you shall receive and protocol header data.

 @param lpBuffer Long pointer to a buffer that receives data. 
 @param nNumberOfBytesToRead Specifies length in bytes of data to be read.   
 
 @return the length of bytes that were successfully received from printer port if there are
 no errors and negative value if there is an error. The particular return value depends on the
 type of error. 
*/
int PDSDK_ReadPortDirect(LPVOID lpBuffer, int nNumberOfBytesToRead);

/**
 This function cause write data direct to printer.
 
 If printer is into protocol mode this function send protocol packet header data.

 @param lpBuffer Long pointer to a buffer that contains data to be write. 
 @param nNumberOfBytesToWrite Specifies length in bytes of data to be write.   
 
 @return the length of bytes that were successfully write to printer port if there are no 
 errors and negative value if there is an error. The particular return value depends 
 on the type of error.  
*/
int PDSDK_WritePrinterDirect(LPCVOID lpBuffer, int nNumberOfBytesToWrite);

/**
 This function wait until there is pending data in printer input buffer.  
 
 @note You can use that function to ensure that printer can accept commands immediately.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.     
*/
int PDSDK_Flush();

/**
 This function returns graphics page dimentions.  
 
 @param lpszPage Long pointer to a buffer that receives graphics page dimentions.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.     
*/
int PDSDK_GetGraphicsPageSize(LPSIZE lpszPage);

/**
 This function starts a graphics print job.  
 
 @note Driver does not require valid document name and don't handle it.
 
 @param lpszDocName Long pointer to buffer that constains document name. If value is NULL then 
 document name is ignore. 
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.      
*/
int PDSDK_StartDoc(LPCTSTR lpszDocName);

/**
 This function ends a graphics print job. 
  
 @param bAbort Indicate to abort the current graphics print job.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.      
*/
int PDSDK_EndDoc(BOOL bAbort);

/*
 This function prepares the printer driver to accept graphics data.   
 
 Must call this function every time when you need to print new graphics page. Dimentions of that 
 page can be obtain with PDSDK_GetGraphicsPageSize. All data outside that dimentions shall be discard.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.      
*/
int PDSDK_StartPage(void);

/**
 This function send graphics data to printer. 
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.      
*/
int PDSDK_EndPage(void);

/**
 This function cause print self test. 
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.      
*/
int PDSDK_SelfTest(void);

/**
 This function read printer identification string. 
  
 Data in range [0 .. 21] contains printer name.
 Data in range [22 .. 26] contains printer firmware version.      

 @param lpBuffer Long pointer to a buffer that receives data. It must be 32 bytes long. 
 @param nBufferSize Specifies the size of buffer in bytes.
 
 @return the length of bytes that was successfully received from printer if there are no errors 
 and negative value if there is an error. The particular return value depends on the type of error.  
*/
int PDSDK_ReadIdentification(LPBYTE lpBuffer, int nBufferSize);

/**
 This function read printer battery voltage level and temperature of head.
  
 @param lpnVoltage Long pointer to a variable that receives battery voltage level. Voltage level 
 must be devide by 10 to receive value in volts.
 @param lpnTemperature Long pointer to a variable that receives temperature of printer head.  
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.   
*/
int PDSDK_ReadVoltageAndTemperature(int *lpnVoltage, int *lpnTemperature);

/* Define status bits. */
#define	PDSDK_PRINTERSTATUS_NOPAPER  0x04 /* Printer is out of paper. */
/* Define status bits definitions. */

/**
 This function read printer status.
 
 @param lpnStatus Long pointer to a variable that receives printer status.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.   
*/
int PDSDK_ReadStatus(int *lpnStatus);

/**
 This functions read printer serial number.

 @note This function is left only for compatibility with older firmware.

 @param lpBuffer Long pointer to a buffer that receives NULL terminated printer serial number. It must
 be at least 14 bytes long.

 @param nBufferSize Specifies the size of buffer in bytes.    
 
 @return the length of bytes that was successfully received from printer if there are no errors 
 and negative value if there is an error. The particular return value depends on the type of error.   
*/
int PDSDK_ReadSerialNumber(LPBYTE lpBuffer, int nBufferSize);


/**
 This functions read printer serial number.

 @param lpBuffer Long pointer to a buffer that receives NULL terminated printer serial number. It must
 be at least 14 bytes long.

 @param nBufferSize Specifies the size of buffer in bytes.    
 
 @return the length of bytes that was successfully received from printer if there are no errors 
 and negative value if there is an error. The particular return value depends on the type of error.   
*/
int PDSDK_ReadSerialNumberEx(LPBYTE lpBuffer, int nBufferSize);

/**
 This function cause print text.
   
 You can insert tags into text to change its parameters.
 Valid tags are:
 [BEL]     - Bel.
 [HT]      - Horizontal tab.
 [LF]      - Line feed.
 [U+]      - Start underline.
 [U-]      - End underline.
 [I+]      - Start italic.
 [I-]      - End italic.
 [B+]      - Start highlight.
 [B-]      - End highlight.
 [FI]      - Select internal fontset.
 [FL]      - Select loadable fontset.
 [FA]      - Select font A.
 [FB]      - Select font B.
 [DW+]     - Start double width text.
 [DW-]     - End double width text.
 [DH+]     - Start double height text.
 [DH-]     - End double height text.
 [FV+]     - Flip chars vertical.
 [FV-]     - End flip chars vertical.
 [RC+]     - Start rotate characters right (90 degrees).
 [RC-]     - End rotate characters right (90 degrees).
 [ILxxx]   - Set intensity level: IL70, IL80, IL90, IL100, IL120, IL150.
 [LEFT]	 - Align text to left. Valid only to next new line.
 [CENTER]	 - Align text to center. Valid only to next new line.
 [RIGHT]	 - Align text to right. Valid only to next new line.
 @param lpText Long pointer to buffer that contains text.  
 @param nTextLength Specifies the buffer length.   
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.   
*/
int PDSDK_PrintTaggedText(LPCSTR lpText, int nTextLength);

/* Define magstripe reader parameters. */
#define	PDSDK_MRPARAM_TRACK1	0x01 /* Read Track 1. */
#define	PDSDK_MRPARAM_TRACK2	0x02 /* Read Track 2. */
#define	PDSDK_MRPARAM_TRACK3	0x04 /* Read Track 3. */
#define	PDSDK_MRPARAM_PREFIX	0x08 /* Delimited track data with
                                      * 0xF1, 0xF2, and 0xF3. */
#define	PDSDK_MRPARAM_RAW		0x40 /* Return data in hex string. */
/* End of magstripe reader definition. */

/**
 This function read magnetic card.
  
 @param nTimeout Specifies timeout of operation.

 @param nParam Specifies parameters for reading magstripe. 
 @param lpBuffer Long pointer to a buffer that receives magstripe data. 
 @param nBufferSize Specifies the size of buffer in bytes.
 
 @return the length of bytes that was successfully received from printer if there are no errors 
 and negative value if there is an error. The particular return value depends on the type of error.   
*/
int PDSDK_ReadMagstripe(int nTimeout, int nParam, LPBYTE lpBuffer, int nBufferSize);

/* Payment card structure. */
#pragma pack(1)
typedef struct 
{	
	/* Card data. */
    char acCardNumber[20];  /* 20 symbols with null terminated byte. */
	char acCardName[28];    /* If '/' present it is a separator. */
	int  nExpMonth;		    /* Expire date - Month. */
	int  nExpYear;		    /* Expire date - Year. */
	/* Track 1 data & size if present. */
	unsigned char ucTrack1[256]; 
	int nTrack1Len;					
	/* Track 2 data & size if present. */
	unsigned char ucTrack2[256];
	int nTrack2Len;
	/* Track 3 data & size if present. */
	unsigned char ucTrack3[256];
	int nTrack3Len;
} 
PDSDK_PAYMENTCARD;
#pragma pack()
/* End of type definition. */

/**
 This function read payment card.
  
 @param nTimeout Specifies timeout of operation. 
 @param nParams Specifies parameters for reading magstripe.
 @param lpPaymentCard Long pointer to a buffer that receives payment card data. Before calling
 function you must specifies which tracks to read.
   
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.   
*/
int PDSDK_ReadPaymentCard(int nTimeout, int nParams, PDSDK_PAYMENTCARD *lpPaymentCard);

/**
 This function read barcode.
  
 @param nTimeout Specifies timeout of operation.  
 @param lpBuffer Long pointer to a buffer that receives barcode data.
 @param nBufferSize Specifies the size of buffer in bytes.
 @param lpszBarcodeName Long pointer to a buffer that receives barcode name. This parameter can be NULL.
 @param nMaxChars Specifies the size, in characters, of the buffer pointed to by the "lpszBarcodeName" parameter.
 
 @return the length of bytes that was successfully received from printer if there are no errors 
 and negative value if there is an error. The particular return value depends on the type of error. 
*/
int PDSDK_ReadBarcode(int nTimeout, LPBYTE lpBuffer, int nBufferSize, LPTSTR lpszBarcodeName, int nMaxChars);

/* Define printed barcode types. */
#define PDSDK_BARCODE_UPCA			0
#define PDSDK_BARCODE_UPCE			1
#define PDSDK_BARCODE_EAN13			2
#define PDSDK_BARCODE_EAN8			3
#define PDSDK_BARCODE_CODE39		4
#define PDSDK_BARCODE_ITF			5
#define PDSDK_BARCODE_CODABAR		6
#define PDSDK_BARCODE_CODE93		7
#define PDSDK_BARCODE_CODE128		8
#define PDSDK_BARCODE_PDF417		9
#define	PDSDK_BARCODE_CODE128AUTO	10
#define	PDSDK_BARCODE_EAN128AUTO	11
/* End of printed barcode types definitions. */

/* Define printed barcode text positions. */
#define PDSDK_BARCODE_NO      0 /* No text printed. */
#define PDSDK_BARCODE_ABOVE   1 /* Text is above barcode. */
#define PDSDK_BARCODE_BELOW   2 /* Text is below barcode. */
#define PDSDK_BARCODE_BOTH    3 /* Text is above and below barcode. */
/* End of printed barcode text positions definitions. */

/* Define printed barcode text fonts. */
#define PDSDK_BARCODE_FONTA   0 /* Font A. */
#define PDSDK_BARCODE_FONTB   1 /* Font B. */
/* End of printed barcode text fonts definitions. */

/* Define printed barcode text aligns. */
#define PDSDK_BARCODE_LEFT        0 /* Text is left align. */
#define PDSDK_BARCODE_CENTER      1 /* Text is center align. */
#define PDSDK_BARCODE_RIGHT       2 /* Text is right align. */
#define PDSDK_BARCODE_LEFTVERT    4 /* Text is left vartical align. */
#define PDSDK_BARCODE_CENTERVERT  5 /* Text is vertical center align. */
#define PDSDK_BARCODE_RIGHTVERT   6 /* Text is vertical right align. */
/* End of printed barcode text aligns definitions. */

/* Printed barcode parameters structure. */
#pragma pack(1)
typedef struct
{	
	int nType;         /* Barcode type. */	
	int nHorzSize;     /* Barcode horizontal size. */	
	int nHeight;       /* Barcode height. */
	int nTextPosition; /* Barcode text position. */	
	int nTextFont;     /* Barcode text font. */	
	int nTextAlign;    /* Barcode text align. */
}
PDSDK_PRINTBARCODE_PARAMS;
#pragma pack()
/* End of type definition. */

/**
 This function cause print barcode.
  
 @param lpProperties Long pointer to structure that specifies parameters of printed barcode. 
 @param lpBarcodeData Long pointer to a buffer that contains barcode data.
 @param nBarcodeDataLength Specifies the barcode data length in bytes.
   
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_PrintBarcode(PDSDK_PRINTBARCODE_PARAMS *lpProperties, LPBYTE lpBarcodeData, int nBarcodeDataLength);

/**
 This function read barcode reader settings.
  
 You can read more then one velue at once.
 
 Example for read parameter:
    DWORD param[2] = 
    {
 	  0x0F, // Parameter type
        0x00  // Reserved for parameter value
    };
    ...
    PDSDK_ReadBarcodeSettings(param, 1);

 @param lpDWordArray Long pointer to a DWORD array that contains requested barcode paramer.
 The elements of that must be thread as multiples of two DWORD - one for parameter and next for value.
 @param nNodes Specifies how many couples of DWORD are pointed by the "lpDWordArray" parameter. 
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int	PDSDK_ReadBarcodeSettings(LPDWORD lpDWordArray, int nNodes);

/**
 This function write barcode reader settings.
  
 You can write more then one value at once.
 
 Example for write parameter:
    DWORD param[2] = 
    {
 	  0x0F, // Parameter type
        0x05  // Parameter value
    };
    ...
    PDSDK_WriteBarcodeSettings(param, 1);  

 @param lpDWordArray Long pointer to a DWORD array that contains requested barcode paramer.
 The elements of that must be thread as multiples of two DWORD - one for parameter and next for value.
 @param nNodes Specifies how many couples of DWORD are pointed by the "lpDWordArray" parameter.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int	PDSDK_WriteBarcodeSettings(LPDWORD lpDWordArray, int nNodes);

/**
 This function cause turn off printer.   
 
 @note This function make a call to PDSDK_ClosePrinter.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_TurnOff(void);

/**
 This function cause paper feed.
 
 @param nLines Specifies how many pixels to feed paper. Eigth pixels are one millimeter (203 DPI). If value 
 is 0 then paper is feed from printer head to the cutter.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_PaperFeed(int nLines);

/* Defines logo parameters. */
#define PDSDK_LOGO_NORMAL        0
#define PDSDK_LOGO_DOUBLEWIDE    1
#define PDSDK_LOGO_DOUBLEHIGH    2
#define PDSDK_LOGO_DOUBLEBOTH    (PDSDK_LOGO_DOUBLEWIDE + \
	                              PDSDK_LOGO_DOUBLEHIGH)
/* End of logo parameters definitions. */

/**
 This function cause logo printing.
 
 @param nLogoParams Specifies logo parameter.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_PrintLogo(int nLogoParams);

/**
 This function cause logo loading into printer.
 
 @param hLogoImage Handle to bitmap that contains logo.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_LoadLogoFromBitmap(HBITMAP hLogoImage);

/**
 This function set line feed rate of minimum pitch.
 
 @param nLines Specifies the line feed rate per line is specified by n/203 inch. If value is 0 then
 function set default line feed rate per line is specified by 1/6 inch.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_SetLineFeedRate(int nLines);

/**
 This function do black mark mode calibration.  
 
 The printer moves approximately 20 cm of paper and searches minimum and maximum of paper sensor values. 
 Then a reasonable value of the ADC threshold is calculated and stored in flash memory.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_BMMCalibration();

/**
 This function print bitmap.  
 
 @param hBitmap Specifies handle to bitmap.
 @param nDstX Specifies the x-coordinate, in pixels, of the upper left corner of the destination rectangle.
 @param nDstY Specifies the y-coordinate, in pixels, of the upper left corner of the destination rectangle.
 @param nDstWidth Specifies the width, in pixels, of the destination rectangle.
 @param nDstHeight Specifies the height, in pixels, of the destination rectangle.
 @param nSrcX Specifies the x-coordinate, in pixels, of the upper left corner of the source rectangle.
 @param nSrcY Specifies the y-coordinate, in pixels, of the upper left corner of the source rectangle.
 @param nSrcWidth Specifies the width, in pixels, of the source rectangle.
 @param nSrcHeight Specifies the height, in pixels, of the source rectangle.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_PrintBitmap(HBITMAP hBitmap, int nDstX, int nDstY, int nDstWidth, int nDstHeight,	
					  int nSrcX, int nSrcY, int nSrcWidth, int nSrcHeight);

/* Defines codetable. */
#define PDSDK_CODETABLE_ENGLISH		0	// ENGLISH (437)
#define PDSDK_CODETABLE_LATIN1		1	// LATIN 1 (850)
#define PDSDK_CODETABLE_PORTUGUESE	2	// PORTUGUESE (860)
#define PDSDK_CODETABLE_LITHUANIAN	3	// LITHUANIAN
#define PDSDK_CODETABLE_LATIN2		4	// LATIN 2 (852)
#define PDSDK_CODETABLE_POLISH		5	// POLISH
#define PDSDK_CODETABLE_TURKISH857	6	// TURKISH (857)
#define PDSDK_CODETABLE_BALTIC775	7	// BALTIC (775)
#define PDSDK_CODETABLE_BULGARIAN	8	// BULGARIAN (856)
#define PDSDK_CODETABLE_RUSSIAN		9	// RUSSIAN (866)
#define PDSDK_CODETABLE_LATVIAN		10	// LATVIAN
#define PDSDK_CODETABLE_GREEK737	11	// GREEK (737)
#define PDSDK_CODETABLE_HEBREW862	12	// HEBREW (862)
#define PDSDK_CODETABLE_WESTERN		13	// WESTERN (1252)
#define PDSDK_CODETABLE_CE			14	// CE (1250)
#define PDSDK_CODETABLE_TURKISH1254	15	// TURKISH (1254)
#define PDSDK_CODETABLE_BALTIC1257	16	// BALTIC (1257)
#define PDSDK_CODETABLE_CYRILLIC	17	// CYRILLIC (1251)
#define PDSDK_CODETABLE_GREEK1253	18	// GREEK (1253)
#define PDSDK_CODETABLE_HEBREW1255	19	// HEBREW (1255)
#define PDSDK_CODETABLE_KATAKANA	20	// KATAKANA 
#define PDSDK_CODETABLE_ARABIC1		21	// ARABIC
#define PDSDK_CODETABLE_ARABIC2		22	// ARABIC (1256)
#define PDSDK_CODETABLE_ARABIC3		23	// ARABIC (1256 with Arabic digits and punctuation)
#define PDSDK_CODETABLE_ARABIC4		24	// ARABIC (1256 with Farsi Arabic digits and punctuation)
/* End of code table definitions. */

/**
 This function select code table.  
 
 @param nCodetable Specifies code table.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_SelectCodetable(int nCodetable);

/**
 This function set the length of label.  
 
 @param nLength Length of the printing area.
 @param nGap Length of gap between labels.
 @param nFeed Feed amount for the beginning of a line after label feed.
 @param nBackward Feed back amount before the beginning of printing.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_SetLabel(int nLength, int nGap, int nFeed, int nBackward);

/**
 This function feed label.  
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_LabelFeed();

/* Defines channels. */
#define PDSDK_CHANNEL_PRINTER       0x01 /* Printer. */
#define PDSDK_CHANNEL_SMARTCARD	    0x02 /* Smart Card Reader. */
#define PDSDK_CHANNEL_GPRSMODEM     0x05 /* GPRS modem. */
#define PDSDK_CHANNEL_MIFARE        0x10 /* Mifare Card reader. */
#define PDSDK_CHANNEL_RFID          0x0D /* RFID Contactless Card reader. */
/* End of channel definitions. */

/**
 This function cause open channel with external module.
 
 @note Printer channel is always opened.

 @param nChannel Specifies the channel. 
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.  
*/
int PDSDK_OpenChannel(int nChannel);

/**
 This function cause close channel with external module.
 
 @note Printer channel can not be closed.

 @param nChannel Specifies the channel. 
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.  
*/
int PDSDK_CloseChannel(int nChannel);

/**
 This function cause sending data from specific channel.
 
 @param nChannel Specifies the channel.  
 @param lpBuffer Long pointer to a buffer that constains data to be send.
 @param nNumberOfBytesToSend Specifies length in bytes of data to be send.
 
 @return If no error occurs, this function returns number of bytes sended. If an error occurs, a negative value
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_SendToChannel(int nChannel, LPBYTE lpBuffer, int nNumberOfBytesToSend);

/**
 This function cause receiving data from specific channel.
 
 @param nChannel Specifies the channel.  
 @param lpBuffer Long pointer to a buffer that receives data.
 @param nBufferSize Specifies length in bytes of data to be received.
 
 @return If no error occurs, this function returns number of bytes received. If an error occurs, a negative value
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_RecvFromChannel(int nChannel, LPBYTE lpBuffer, int nBufferSize);

/**
 This function cause receiving data from specific channel.
 
 @param nChannel Specifies the channel.  
 @param nSpeed Specifies the channel speed. 
   0 Sets speed to 1200 bps.
   1 Sets speed to 2400 bps.
   2 Sets speed to 4800 bps.
   3 Sets speed to 9600 bps.
   4 Sets speed to 19200 bps.
   5 Sets speed to 57600 bps.
   6 Sets speed to 115200 bps. 
 @param nFlowControl Specifies the channel flow control type.
   0 Sets no flow control.
   1 Sets XON/XOFF flow control.
   2 Sets RTS/CTR flow control.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.  
*/
int PDSDK_SetChannel(int nChannel, int nSpeed, int nFlowControl);

/* ************************************************************************************************
 * Next functions are related with GPS module. 
 * ************************************************************************************************
*/

/**
 This function receive NMEA message.
 
 @param nTimeout Specifies timeout of operation.
 @param lpBuffer Long pointer to a buffer that receives message data.
 @param nBufferSize Specifies the size of buffer in bytes.
 
 @return The number of bytes bytes that was successfully received from GPS module if there are no errors 
 and negative value if there is an error. The particular return value depends on the type of error.
*/
int PDSDK_GPS_ReceiveNMEA(int nTimeout, LPBYTE lpBuffer, int nBufferSize);

/* ************************************************************************************************
 * Next functions are related with universal reader. 
 * ************************************************************************************************
*/
/* Declare univarsal reader status bytes constants. */
#define	PDSDK_STATUS1_ERROR       0x01 /* Error. */
#define	PDSDK_STATUS1_BTNPRESSED  0x02 /* Button is pressed. */
#define	PDSDK_STATUS1_ANSWEROFBTN 0x04 /* Auto answear. */
#define	PDSDK_STATUS1_MULTIFRAME  0x08 /* Part of multiframe. */

#define	PDSDK_STATUS2_0K		  0x00 /* Successful. */
#define	PDSDK_STATUS2_INVALIDCMD  0x01 /* Invalid command. */
#define	PDSDK_STATUS2_NOTPERMITED 0x02 /* Not permited command. */
#define	PDSDK_STATUS2_CARDERROR   0x03 /* Card error. */
#define	PDSDK_STATUS2_SYSTEMERROR 0x04 /* System error. */
#define	PDSDK_STATUS2_NORESPONCE  0x05 /* No responce. */
#define	PDSDK_STATUS2_COMMERROR   0xFF /* Communication error. */
/* End of univarsal reader status bytes constants. */

/**
 This function returns status bytes of universal reader.
 
 @param nChannel Specifies requested channel. 
 @param lpnStatus1 Long pointer to a variable that receives status1. 
 @param lpnStatus2 Long pointer to a variable that receives status2.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.  
*/
int PDSDK_UniversalReader_GetStatus(int nChannel, int *lpnStatus1, int *lpnStatus2);

/**
 This function receives identification string of universal reader. 
  
 Data in range [0 .. 21] contains universal reader name.
 Data in range [22 .. 26] contains universal reader firmware version.

 @param nChannel Specifies requested channel.
 @param lpBuffer Long pointer to a buffer that receives identification string. It's size may vary 
 depends of device.
 @param nBufferSize Specifies the size of buffer in bytes. 
 
 @return The number of bytes that were successfully received from universal reader if there are no errors
 and negative value if there is an error. The particular return value depends on the type of error.  
*/
int PDSDK_UniversalReader_ReadIdentification(int nChannel, LPBYTE lpBuffer,  int nBufferSize);

/* ************************************************************************************************
 * Next functions are related with smart card reader.
 * Before using any of them you must open smart card reader channel.
 * Please look at PDSDK_OpenChannel how to do that.
 *
 * You can get status bytes of the smart card reader with 
 * PDSDK_UniversalReader_GetStatus.
 *
 * You can read identification string of smart card reader with 
 * PDSDK_UniversalReader_ReadIdentification
 *
 * This functions works only in protocol mode.
 * ************************************************************************************************
*/

/**
 This function cause power down of smart card reader.  
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.  
*/
int PDSDK_SCR_PowerOff(void);

/**
 This function cause power on of smart card reader.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.  
*/
int PDSDK_SCR_PowerOn(void);

/**
 This function check if there is card available in reader.
 
 @params lpbPresent Long pointer to a variable that receives card availability.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.  
*/
int PDSDK_SCR_IsCardAvailable(INT_BOOL *lpbPresent);

/* Defines smart card types. */
#define PDSDK_SCTYPE_MCU  0x01 /* MCU card. */
/* End of smart card types definitions. */

/**
 This function set card type.
 
 @params nCardType Specifies card type.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_SCR_SetCardType(int nCardType);

/* Defines smart card reader slot types. */
#define PDSDK_SCRSLOT_MCU  0x00 /* MCU card. */
#define PDSDK_SCRSLOT_SAM  0x01 /* SAM module. */
/* End of smart card reader slot rypes definitions. */

/**
 This function set card slot type.
 
 @param nCardSlotType Specifies slot type.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.  
*/
int PDSDK_SCR_SetCardSlotType(int nCardSlotType);

/**
 This function cause smart card reset.
 
 @param lpBuffer Long pointer to a buffer that receives data after smart card reset.
 @param nBufferSize Specifies the size of buffer in bytes.
 
 @return the number of bytes that was successfully received from smart card reader if there are no 
 errors and negative value if there is an error. The particular return value depends on the type of error. 
*/
int PDSDK_SCR_MCU_Reset(LPBYTE lpBuffer, int nBufferSize);

/**
 This function cause execution of smart card ADPU OUT command.
 
 @param lpDataIn Long pointer to a buffer that contains command data to be send to smart card.
 @param nDataInSize Specifies the size of command data in bytes.
 @param lpDataOut Long pointer to a buffer that receives data after command execution.
 @param nDataOutSize Specifies the size of buffer in bytes.
 
 @return The number of bytes that were successfully received from smart card reader if there are no 
 errors and negative value if there is an error. The particular return value depends on the type of error. 
*/
int PDSDK_SCR_MCU_ADPU_OUT(const LPBYTE lpDataIn, int nDataInSize, LPBYTE lpDataOut, int nDataOutSize);

/**
 This function cause execution of smart card ADPU IN command.
 
 @param lpDataIn Long pointer to a buffer that contains command data to be send to smart card.
 @param nDataInSize Specifies the size of command data in bytes.
 @param lpDataOut Long pointer to a buffer that receives data after command execution.
 @param nDataOutSize Specifies the size of buffer in bytes.
 
 @return The number of bytes that were successfully received from smart card reader if there are no 
 errors and negative value if there is an error. The particular return value depends on the type of error. 
*/
int PDSDK_SCR_MCU_ADPU_IN(const LPBYTE lpDataIn, int nDataInSize, LPBYTE lpDataOut, int nDataOutSize);

/**
 This function cause execution of smart card C ADPU command.
 
 @param lpDataIn Long pointer to a buffer that contains command data to be send to smart card.
 @param nDataInSize Specifies the size of command data in bytes.
 @param lpDataOut Long pointer to a buffer that receives data after command execution.
 @param nDataOutSize Specifies the size of buffer in bytes.
 
 @return The number of bytes that were successfully received from smart card reader if there are no 
 errors and negative value if there is an error. The particular return value depends on the type of error. 
*/
int PDSDK_SCR_MCU_C_ADPU(const LPBYTE lpDataIn, int nDataInSize, LPBYTE lpDataOut, int nDataOutSize);

/* ****************************************************************************
 * Next functions are related with mifare card reader.
 * Before using any of them you must open mifare card reader channel.
 * Please look at PDSDK_OpenChannel how to do that.
 *
 * You can get status bytes of the mifare card reader with 
 * PDSDK_UniversalReader_GetStatus.
 *
 * You can read identification string of mifare card reader with 
 * PDSDK_UniversalReader_ReadIdentification
 *
 * This functions works only in protocol mode.
 * ****************************************************************************
*/

/**
 This function cause power on of mifare card reader.  
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_PowerOn(void);

/**
 This function cause power down of mifare card reader.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_PowerOff(void);

/**
 This function configure mifare card reader.  
 
 @param pStatus Long pointer to a variable that receives status from function execution.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_Config(LPBYTE pStatus);

/* Define mifare request types. */
#define PDSDK_MIFARE_RQTYPE_IDLECARDS 0x00 /* Idle cards. */
#define PDSDK_MIFARE_RQTYPE_ALLCARDS  0x01 /* All cards. */
/* End of mifare request types definitions. */

/**
 This function cause request for mifare card.
 
 @param nRequestType Specifies request type.
 @param pStatus Long pointer to a variable that receives status from function execution.
 @param pRq1 Long pointer to a variable that receives RQ1. 
 @param pRq2 Long pointer to a variable that receives RQ2.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_Request(int nRequestType, LPBYTE pStatus, LPBYTE pRq1, LPBYTE pRq2);

/**
 This function cause anticollision for mifare card.
 
 @param pStatus Long pointer to a variable that receives status from function execution.
 @param pnSerialNumber Long pointer to a variable that receives mifare card serial number.  
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_Anticollision(LPBYTE pStatus, UINT *pnSerialNumber);

/**
 This function cause selection of mifare card.
 
 @param nSerialNumber Specifies mifare card serial number. 
 @param pStatus Long pointer to a variable that receives status from function execution.
 @param pSAck Long pointer to a variable that receives SACK value.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_SelectCard(UINT nSerialNumber, LPBYTE pStatus, LPBYTE pSAck);

/* Define mifare authentication key types. */
#define PDSDK_MIFARE_AUTHKEYTYPE_KEYA	0x60 /* Key A. */
#define PDSDK_MIFARE_AUTHKEYTYPE_KEYB	0x61 /* Key B. */
/* End of authentication key types definition. */

/**
 This function authenticate block with direct supplied key.
 
 @param nSerialNumber Specifies mifare card serial number.
 @param nAuthKeyType Specifies authentication key type.
 @param nBlockNumber Specifies requested block number for authentication.
 @param ucKey Long pointer to a buffer that contains the six bytes authentication key.
 @param pStatus Long pointer to a variable that receives status from function execution.  
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_AuthByKey(UINT nSerialNumber, int nAuthKeyType, int nBlockNumber, const BYTE ucKey[6], LPBYTE pStatus);

/**
 This function read data from mifare card block.
 
 @param nBlockNumber Specifies mifare card block number.
 @param pStatus Long pointer to a variable that receives status from function execution.
 @param lpDataOut Long pointer to a buffer that receives 16 bytes data.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_Read(int nBlockNumber, LPBYTE pStatus, BYTE lpDataOut[16]);

/**
 This function write data to mifare card block.
 
 @param nBlockNumber Specifies mifare card block number.
 @param lpDataIn Long pointer to a buffer that contains 16 bytes data to write.
 @param pStatus Long pointer to a variable that receives status from function execution.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_Write(int nBlockNumber, const BYTE lpDataIn[16], LPBYTE pStatus);

/**
 This function increment value in mifare card.
 
 @param nSrcBlockNumber Specifies source block number.
 @param nDstBlockNumber Specifies destination block number.
 @param nValue Specifies increment value.
 @param pStatus Long pointer to a variable that receives status from function execution.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_ValueIncrement(int nSrcBlockNumber, int nDstBlockNumber, int nValue, LPBYTE pStatus);

/**
 This function decrement value in mifare card.
 
 @param nSrcBlockNumber Specifies source block number.
 @param nDstBlockNumber Specifies destination block number.
 @param nValue Specifies decrement value.
 @param pStatus Long pointer to a variable that receives status from function execution.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. 
*/
int PDSDK_Mifare_ValueDecrement(int nSrcBlockNumber, int nDstBlockNumber, int nValue, LPBYTE pStatus);

/**
 This function restore value in mifare card.
 
 @param nSrcBlockNumber Specifies source block number.
 @param nDstBlockNumber Specifies destination block number.
 @param nValue Specifies restore value.
 @param pStatus Long pointer to a variable that receives status from function execution.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_Mifare_ValueRestore(int nSrcBlockNumber, int nDstBlockNumber, int nValue, LPBYTE pStatus);

/**
 This function cause halt of mifare card.
  
 @param pStatus Long pointer to a variable that receives status from function execution.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_Mifare_Halt(LPBYTE pStatus);

/**
 This function get mifare module serial number.
  
 @param pStatus Long pointer to a variable that receives status from function execution.
 @param pnSerialNumber Long pointer to a variable that receives module serial number.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_Mifare_GetSerial(LPBYTE pStatus, UINT *pnSerialNumber);

/**
 This function write data in EEPROM of mifare module.
  
 @param nAddress Specifies EEPROM address.
 @param lpBuffer Long pointer to a buffer that constains data to be write.
 @param nBufferSize Specifies the size in bytes of data to be write.
 @param pStatus Long pointer to a variable that receives status from function execution.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_Mifare_WriteEEPROM(int nAddress, const LPBYTE lpBuffer, int nBufferSize, LPBYTE pStatus);

/**
 This function write four bytes data to mifare card block.
  
 @param nBlockNumber Specifies block number.
 @param nValue Specifies value to be write.
 @param pStatus Long pointer to a variable that receives status from function execution.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_Mifare_WriteValue(int nBlockNumber, int nValue, LPBYTE pStatus);

/**
 This function load key into mifare card reader memory.
  
 @param nSelector Specifies selector. Selector is a position in memory.
 @param lpKey Long pointer to a buffer that contains six bytes key.
 @param pStatus Long pointer to a variable that receives status from function execution.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_Mifare_LoadKey(int nSelector, const BYTE lpKey[6], LPBYTE pStatus);

/**
 This function authenticate mifare card block with key from mifare reader memory.
  
 @param nAuthKeyType Specifies authentication key type. 
 @param nSerialNumber Specifies mifare card serial number.
 @param nSelector Specifies selector.
 @param nBlockNumber Specifies mifare card block number.
 @param pStatus Long pointer to a variable that receives status from function execution.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
*/
int PDSDK_Mifare_AuthByEEPROM(int nAuthKeyType, UINT nSerialNumber, int nSelector, int nBlockNumber, LPBYTE pStatus);

/**
 This function read data from mifare reader EEPROM.
  
 @param nAddress Specifies EEPROM address.
 @param pStatus Long pointer to a variable that receives status from function execution.  
 @param lpBuffer Long pointer to a buffer that receives data. 
 @param nBufferSize Specifies the size of buffer in bytes.
  
 @return The number of bytes that were successfully received from mifare reader EEPROM if there are no errors and negative
 value if there is an error. The particular return value depends on the type of error.
*/
int Mifare_ReadEEPROM(int nAddress, LPBYTE pStatus,LPBYTE lpBuffer, int nBufferSize);


/* ************************************************************************************************
 * Next functions are related with printer page mode. In page mode the result 
 * of incoming commands is forwarded to a reserved memory area (page). The page 
 * place and size is defined using function PDSDK_PageMode_SetWorkingArea. 
 * Function PDSDK_PageMode_SetOrientation selects the print direction in this 
 * page. At the end the collected information is printed only in the area between 
 * the first and last line containing at least one black point.
 * Almost all other functions works in page mode. The centering and right 
 * alignment (PDSDK_PrintTaggedText) is working in the currently defined page 
 * width.
 * ************************************************************************************************
*/

/**
 This function start page mode.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. Function may return error and if 
 printer does not support page mode (PDSDK_EFAILED).
*/
int PDSDK_PageMode_Start();

/**
 This function returns printer in standart mode.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. Function may return error and if 
 printer does not support page mode (PDSDK_EFAILED).
*/
int PDSDK_PageMode_End();

/**
 This function cause printing of area between the first and last line containing at least one black point. 
 The content if not clear so you may print it many times.
  
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. Function may return error and if 
 printer does not support page mode (PDSDK_EFAILED).
*/
int PDSDK_PageMode_Print();

/**
 This function returns the maximum printable area size.
  
 @param pszArea Long pointer to buffer that receives area size.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. Function may return error and if 
 printer does not support page mode (PDSDK_EFAILED).
*/
int PDSDK_PageMode_GetGlobalArea(SIZE *pszArea);

/* Define page mode directions types. */
#define PAGEMODE_DIRECTION_LTRB  0	// Printing from left to right, feed to bottom. Starting point in left top corner of the page.
#define PAGEMODE_DIRECTION_LBRT  1	// Printing from bottom to top, feed to right. Starting point in left bottom corner of the page.
#define PAGEMODE_DIRECTION_RBLT  2	// Printing from right to left, feed to top. Starting point in right bottom corner of the page.
#define PAGEMODE_DIRECTION_RTLB  3	// Printing from top to bottom, feed to left. Starting point in right top corner of the page.
/* End of page mode directions definition. */

/**
 This function set working area into global page area.

 @note The command defines the relative position and size of the page.
  
 @param nX Specifies X position of rectangle.
 @param nY Specifies Y position of rectangle.
 @param nWidth Specifies width of rectangle.  
 @param nHeight Specifies height of rectangle. 
 @param nDirection Specifies direction.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. Function may return error and if 
 printer does not support page mode (PDSDK_EFAILED).  
*/
int PDSDK_PageMode_SetWorkingArea(int nX, int nY, int nWidth, int nHeight, int nDirection);

/**
 The function set position in working area.  
 
 @note Don't you this command right before PDSDK_PrintBitmap. If you need 
 to draw bitmap on the specific position use PDSDK_PageMode_SetWorkingArea 
 instead.

 @param nX Specifies the absolute horizontal position for printing.
 @param nY Specifies the absolute vertical position for printing.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. Function may return error and if 
 printer does not support page mode (PDSDK_EFAILED).   
*/
int PDSDK_PageMode_SetPosition(int nX, int nY);

/* Define page mode color modes types. */
#define PAGEMODE_COLOR_WHITE	0
#define PAGEMODE_COLOR_BLACK	1
#define PAGEMODE_COLOR_INVERSE	2
/* End of page mode color modes definition. */

/**
 The function fill or invert a rectangle in page mode.  
  
 @param nX Specifies X position of rectangle.
 @param nY Specifies Y position of rectangle.
 @param nWidth Specifies width of rectangle.  
 @param nHeight Specifies height of rectangle.    
 @param nColor Specifies fill color.  
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. Function may return error and if 
 printer does not support page mode (PDSDK_EFAILED).  
*/
int PDSDK_PageMode_FillRectangle(int nX, int nY, int nWidth, int nHeight, int nColor);

/**
 The function print a rectangular box with selected thickness in page mode.  
  
 @param nX Specifies X position of frame.
 @param nY Specifies Y position of frame.
 @param nWidth Specifies width of frame.  
 @param nHeight Specifies height of frame.  
 @param nPixels Specifies border width in pixels of frame.  
 @param nColor Specifies frame color.  
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error. Function may return error and if 
 printer does not support page mode (PDSDK_EFAILED). 
*/
int PDSDK_PageMode_RectangleFrame(int nX, int nY, int nWidth, int nHeight, int nPixels, int nColor);


/* ************************************************************************************************
 * Next functions are related with RF module. 
 * ************************************************************************************************
*/

/** @defgroup G_LNRFREADER RF Reader Functions
 Functions to work with the RF cards reader
 @{
 */

/**
 ISO14443 Type A (Mifare) cards will be detected
 */
#define PDSDK_CARD_SUPPORT_TYPE_A 0x0001
/**
 ISO14443 Type B cards will be detected. Currently unsupported.
 */
#define PDSDK_CARD_SUPPORT_TYPE_B 0x0002
/**
 Felica cards will be detected.
 */
#define PDSDK_CARD_SUPPORT_FELICA 0x0004
/**
 NFC cards will be detected. Currently unsupported.
 */
#define PDSDK_CARD_SUPPORT_NFC 0x0008
/**
 Jewel cards will be detected. Currently unsupported.
 */
#define PDSDK_CARD_SUPPORT_JEWEL 0x0010
/**
 ISO15693 cards will be detected
 */
#define PDSDK_CARD_SUPPORT_ISO15 0x0020
/**
 ST SRI cards will be detected
 */
#define PDSDK_CARD_SUPPORT_STSRI 0x0040


/**
 RF card types
 */

/**
 Unknown card
 */
#define PDSDK_CARD_UNKNOWN                0
/**
 Mifare Mini
 */
#define PDSDK_CARD_MIFARE_MINI            1
/**
 Mifare Classic 1K
 */
#define PDSDK_CARD_MIFARE_CLASSIC_1K      2
/**
 Mifare Classic 4K
 */
#define PDSDK_CARD_MIFARE_CLASSIC_4K      3
/**
 Mifare Ultralight
 */
#define PDSDK_CARD_MIFARE_ULTRALIGHT      4
/**
 Mifare Ultralight C
 */
#define PDSDK_CARD_MIFARE_ULTRALIGHT_C    5
/**
 ISO 14443A
 */
#define PDSDK_CARD_ISO14443A              6
/**
 Mifare Plus
 */
#define PDSDK_CARD_MIFARE_PLUS            7
/**
 ISO 15693
 */
#define PDSDK_CARD_ISO15693               8
/**
 Mifare Desfire
 */
#define PDSDK_CARD_MIFARE_DESFIRE         9
/**
 ISO 14443B
 */
#define PDSDK_CARD_ISO14443B              10
/**
 FeliCa
 */
#define PDSDK_CARD_FELICA                 11
/**
 ST SRI
 */
#define PDSDK_CARD_ST_SRI                 12

/**
 FeliCa SmartTag battery status
 */

/**
 Normal, card can be used
 */
#define PDSDK_FELICA_SMARTTAG_BATTERY_NORMAL1   0
/**
 Normal, card can be used
 */
#define PDSDK_FELICA_SMARTTAG_BATTERY_NORMAL2   1
/**
 Low, consider replacing
 */
#define PDSDK_FELICA_SMARTTAG_BATTERY_LOW1      2
/**
 Very Low, replace it
 */
#define PDSDK_FELICA_SMARTTAG_BATTERY_LOW2      3

/**
 FeliCa SmartTag battery status
 */
/**
 The area outside of the image will be emptied
 */
#define PDSDK_FELICA_SMARTTAG_DRAW_WHITE_BACKGROUND   0
/**
 The area outside of the image will be shown black
 */
#define PDSDK_FELICA_SMARTTAG_DRAW_BLACK_BACKGROUND   1
/**
 The area outside of the image will be kept to its current state
 */
#define PDSDK_FELICA_SMARTTAG_DRAW_KEEP_BACKGROUND    2
/**
 The area outside of the image will be drawn using one of the previously stored layouts
 */
#define PDSDK_FELICA_SMARTTAG_DRAW_USE_LAYOUT         4

#define PDSDK_MAX_UID_LEN	10

#pragma pack(1)
typedef struct {			
	int index;
	int type;
	unsigned char uid[PDSDK_MAX_UID_LEN];
	int uidLen;
	int capacity;
	int blockSize;
	int maxBlocks;

	union
    {
		struct {
			unsigned short atqa;
			unsigned short atqb;
			unsigned short sak;
		} iso14;
			
		struct {
			unsigned char afi;
			unsigned char dsfid;
		} iso15;      
    };
} 
PDSDK_CARD_INFO;
#pragma pack()

/**
 A call back function that returns card information when it is detected.
 
 @param cardInfo a pointer to the structure that receives card data.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value 
 is returned. The particular return value depends  
 */
typedef void (* PDSDK_RF_CardDetected)(PDSDK_CARD_INFO *cardInfo);

/**
 Initializes and powers on the RF card reader module. Call this function before any other RF card 
 functions. The module power consumption is highly optimized, so it can be left on for extended 
 periods of time.
 
 @param supportedCards any combination of CARD_SUPPORT_* flags to mark which card types to be 
 active. Enable only cards you actually plan to work with, this has high implication on power 
 usage and detection speed. 
 @param callback the PDSDK_RF_CardDetected callback taht is used when card is detected.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value 
 is returned. The particular return value depends on the type of error.  
 */
int PDSDK_RF_Init(int supportedCards, PDSDK_RF_CardDetected callback);

/**
 Powers down RF card reader module. Call this function after you are done with the reader.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value 
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_RF_Close();

/**
 Call this function once you are done with the card, a delegate call rfCardRemoved will be called when
 the card leaves the RF field and new card is ready to be detected.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value 
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_RF_RemoveCard(const PDSDK_CARD_INFO *cardInfo);

/**
 Authenticate mifare card block with direct key data. This is less secure method, as it requires the
 key to be present in the program, the prefered way is to store a key once in a secure environment 
 and then authenticate using the stored key.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param type key type, either 'A' or 'B'.
 @param address the address of the block to authenticate.
 @param key 6 bytes key.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value 
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_MF_AuthByKey(const PDSDK_CARD_INFO *cardInfo, char type, int address, const unsigned char *key);

/**
 Store key in the internal module memory for later use.

 @param keyIndex the index of the key, you can have up to 8 keys stored (0-7).
 @param type key type, either 'A' or 'B'.
 @param key 6 bytes key.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value 
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_MF_StoreKeyIndex(int keyIndex, char type, const unsigned char *key);

/**
 Authenticate mifare card block with previously stored key. This the prefered method, as no key needs 
 to reside in application.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param type key type, either 'A' or 'B'.
 @param address the address of the block to authenticate.
 @param keyIndex the index of the stored key, you can have up to 8 keys stored (0-7).
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value 
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_MF_AuthByStoredKey(const PDSDK_CARD_INFO *cardInfo, char type, int address, int keyIndex);

/**
 Reads one more more blocks of data from Mifare Classic/Ultralight cards. A single read operation gets 
 16 bytes of data, so you can pass 32 on length to read 2 blocks, etc.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param address the address of the block to read.
 @param length the number of bytes to read, this must be multiple of block size (16 bytes).
 @param output the buffer that receives the output data.

 @return If no error occurs, this function returns the number of bytes stored in output buffer. If an 
 error occurs, a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_MF_Read(const PDSDK_CARD_INFO *cardInfo, int address, int length, unsigned char *output);

/**
 Writes one more more blocks of data to Mifare Classic/Ultralight cards. A single write operation 
 stores 16 bytes of data, so you can pass 32 on length to write 2 blocks, etc.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param address the address of the block to write.
 @param data the data to write, must be multiple of the block size (16 bytes).
 @param length the data length in bytes.
 
 @return If no error occurs, this function returns the number of bytes written. If an error occurs, 
 a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_MF_Write(const PDSDK_CARD_INFO *cardInfo, int address, const unsigned char * data, int length);

/**
 Sets the 3DES key of Mifare Ultralight C cards.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param key 16 bytes 3DES key to set.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_MF_UlcSetKey(const PDSDK_CARD_INFO *cardInfo, const unsigned char *key);

/**
 Performs 3DES authentication of Mifare Ultralight C card using the given key

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param key 16 bytes 3DES key to authenticate with.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_MF_UlcAuthByKey(const PDSDK_CARD_INFO *cardInfo, const unsigned char *key);

/**
 Reads one more more blocks of data from ISO 15693 card.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param startBlock the starting block to read from.
 @param length the number of bytes to read, this must be multiple of block size (can be taken
 from the card info that is coming with PDSDK_RF_CardDetected call).
 @param output the buffer that receives the output data.

 @return If no error occurs, this function returns the number of bytes stored in output buffer. If an 
 error occurs, a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_ISO15693_Read(const PDSDK_CARD_INFO *cardInfo, int startBlock, int length, unsigned char *output);

/**
 Writes one more more blocks of data to ISO 15693 card.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param startBlock the starting block to write to.
 @param data the data to write, it must be multiple of block size (can be taken from the card info 
 that is coming with PDSDK_RF_CardDetected call).
 @param length the data length in bytes.

 @return If no error occurs, this function returns the number of bytes written. If an error occurs, 
 a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_ISO15693_Write(const PDSDK_CARD_INFO *cardInfo, int startBlock, const unsigned char * data, int length);

/**
 Reads the security status of one more more blocks from ISO 15693 card.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param startBlock the starting block to read from.
 @param blocks the number of blocks to get the security status.
 @param output the buffer that receives the output data.

 @return If no error occurs, this function returns the number of bytes stored in output buffer. If an 
 error occurs, a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_ISO15693_GetBlocksSecurityStatus(const PDSDK_CARD_INFO *cardInfo, int startBlock, int blocks, unsigned char *output);

/**
 Locks a single ISO 15693 card block. Locked blocks cannot be written upon anymore.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param block the block index to lock.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_ISO15693_LockBlock(const PDSDK_CARD_INFO *cardInfo, int block);

/**
 Changes ISO 15693 card AFI.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param afi new AFI value.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_ISO15693_WriteAFI(const PDSDK_CARD_INFO *cardInfo, unsigned char afi);

/**
 Locks ISO 15693 AFI preventing further changes.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_ISO15693_LockAFI(const PDSDK_CARD_INFO *cardInfo);

/**
 Changes ISO 15693 card DSFID.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param dsfid new DSFID value.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_ISO15693_WriteDSFID(const PDSDK_CARD_INFO *cardInfo, unsigned char dsfid);

/**
 Locks ISO 15693 card DSFID preventing further changes.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.

 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_ISO15693_LockDSFID(const PDSDK_CARD_INFO *cardInfo);

/**
 Reads one more more blocks of data from FeliCa card.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param serviceCode the service code, default is 0x0900.
 @param startBlock the starting block to read from.
 @param length the number of bytes to read, this must be multiple of block size (can be taken from 
 the card info that is coming with PDSDK_RF_CardDetected call).
 @param output the buffer that receives the output data.

 @return If no error occurs, this function returns the number of bytes stored in output buffer. If an 
 error occurs, a negative value is returned. The particular return value depends on the type of error.  
 */
int PDSDK_FELICA_Read(const PDSDK_CARD_INFO *cardInfo, int serviceCode, int startBlock, int length, unsigned char *output);

/**
 Writes one more more blocks of data to FeliCa card.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param serviceCode the service code, default is 0x0900.
 @param startBlock the starting block to write to.
 @param data the data to write, it must be multiple of block size (can be taken from the 
 card info that is coming with PDSDK_RF_CardDetected call).
 @param length the data length in bytes.

 @return If no error occurs, this function returns the number of bytes written. If an error occurs, 
 a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_FELICA_Write(const PDSDK_CARD_INFO *cardInfo, int serviceCode, int startBlock, const unsigned char *data, int length);

/**
 Returns FeliCa SmartTag battery status.

 @note Call this function before any other SmartTag.
 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param status upon successful execution, battery status will be returned here, one of 
 PDSDK_FELICA_SMARTTAG_BATTERY_* constants.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_FELICA_SmartTagGetBatteryStatus(const PDSDK_CARD_INFO *cardInfo, int *status);

/**
 Clears the screen of FeliCa SmartTag.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param status upon successful execution, battery status will be returned here, one of 
 FELICA_SMARTTAG_BATTERY_* constants.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_FELICA_SmartTagClearScreen(const PDSDK_CARD_INFO *cardInfo);

/**
 Draws image on FeliCa SmartTag's screen. The screen is 200x96 pixels.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call. 
 @param x topleft X coordinate in pixels.
 @param y topleft Y coordinate in pixels.
 @param width the width of image.
 @param height the height of image.
 @param drawMode draw mode, one of the FELICA_SMARTTAG_DRAW_* constants.
 @param layout only used when drawMode is PDSDK_FELICA_SMARTTAG_DRAW_USE_LAYOUT, it specifies the index of 
 the layout (1-12) of the previously stored image.
 @param image image data to draw.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_FELICA_SmartTagDrawImage(const PDSDK_CARD_INFO *cardInfo, int x, int y, int width, int height, int drawMode, int layout, const unsigned char *image);

/**
 Saves the current display as layout number.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param layout layout index (1-12) to which the currently displayed image will be saved.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_FELICA_SmartTagSaveLayout(const PDSDK_CARD_INFO *cardInfo, int layout);

/**
 Displays previously stored layout

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param layout layout index (1-12) of the previously stored image.
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_FELICA_SmartTagDisplayLayout(const PDSDK_CARD_INFO *cardInfo, int layout);

/**
 Writes data in FeliCa SmartTag.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param address the address of the card to write to, refer to SmartTag documentation.
 @param data data to write, note that the data does not need to be aligned to block size.
 @param length the data length in bytes.

 @return If no error occurs, this function returns the number of bytes written. If an error occurs, 
 a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_FELICA_SmartTagWrite(const PDSDK_CARD_INFO *cardInfo, int address, const unsigned char *data, int length);

/**
 Writes data in FeliCa SmartTag.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param address the address of the card to read from, refer to SmartTag documentation.
 @param length of the data to read, note that the data does not need to be aligned to block size. 
 @param output the buffer that receives the output data.

 @return If no error occurs, this function returns the number of bytes stored in output buffer. If an 
 error occurs, a negative value is returned. The particular return value depends on the type of error.  
 */
int PDSDK_FELICA_SmartTagRead(const PDSDK_CARD_INFO *cardInfo, int address, int length);

/**
 Waits for FeliCa SmartTag to complete current operation. Waiting is generally not needed, but needed
 in case for example drawing an image and then saving the layout, you need to wait for the image to be
 drawn.
 Write operation forces waiting internally.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call
 
 @return If no error occurs, this function returns PDSDK_ENONE. If an error occurs, a negative value.
 is returned. The particular return value depends on the type of error.
 */
int PDSDK_FELICA_SmartTagWaitCompletion(const PDSDK_CARD_INFO *cardInfo);

/**
 Reads one more more blocks of data from ST SRI card.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param address the starting block to read from.
 @param length the number of bytes to read, this must be multiple of block size (can be taken
 from the card info that is coming with PDSDK_RF_CardDetected call).

 @param output the buffer that receives the output data.

 @return If no error occurs, this function returns the number of bytes stored in output buffer. If an 
 error occurs, a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_STSRI_Read(const PDSDK_CARD_INFO *cardInfo, int address, int length, unsigned char *output);

/**
 Writes one more more blocks of data to ST SRI card.

 @param cardInfo the card information as sent by PDSDK_RF_CardDetected delegate call.
 @param address the starting block to write to.
 @param data the data to write, it must be multiple of block size (can be taken from the card info 
 that is coming with PDSDK_RF_CardDetected call).
 @param length the data length in bytes.

 @return If no error occurs, this function returns the number of bytes written. If an error occurs, 
 a negative value is returned. The particular return value depends on the type of error. 
 */
int PDSDK_STSRI_Write(const PDSDK_CARD_INFO *cardInfo, int address, const unsigned char *data, int length);

/**@}*/

#ifdef __cplusplus
}  /* End of the 'extern "C"' block */
#endif
#endif

