
/****h* General
FUNCTION
  A set of functions for managing all Datecs ESC/POS printers.
*****/
/****h* Magnetic Card
FUNCTION
  Functions related to the magnetic stripe reader module.
*****/
/****h* SmartCard
FUNCTION
  Functions related to the smartcard reader module.
*****/
/****h* Mifare
FUNCTION
  Functions related to the mifare reader module.
*****/
/****h* Barcode
FUNCTION
  Functions related to the barcode reader module.
*****/
/****h* Direct Communication
FUNCTION
  Functions for directly controlling the printer and modules, attached to it
*****/
//typedef int                 BOOL;



#define ERR_OK			0  /**<No error */
#define ERR_TIMEOUT		-6 /**<The device did not respond in a reasonable time. Usualy the device is not connected */
#define ERR_UNSUPPORTED	-7 /**<The device does not support the requested functionality */
#define ERR_BUFFER_TOO_SMALL	-8 /**<The output buffer is not large enough */
#define ERR_INVALID_PARAMS		-9 /**<Some of the parameters has an invalid value */

#define	BAR_UPCA		0
#define	BAR_UPCE		1
#define	BAR_EAN13		2
#define	BAR_EAN8		3
#define	BAR_CODE39		4
#define	BAR_ITF			5
#define	BAR_CODABAR		6
#define	BAR_CODE93		7
#define	BAR_CODE128		8
#define	BAR_PDF417		9

#define ALIGN_LEFT		0
#define ALIGN_CENTER	1
#define ALIGN_RIGHT		2
#define ALIGN_JUSTIFY	3

#define	BAR_SCALE_2		2
#define	BAR_SCALE_3		3
#define	BAR_SCALE_4		4

#define	BAR_TEXT_NONE	0
#define	BAR_TEXT_ABOVE	1
#define	BAR_TEXT_BELOW	2
#define	BAR_TEXT_BOTH	3

#define	BAR_HEIGHT_MIN	1
#define	BAR_HEIGHT_DEFAULT	162
#define	BAR_HEIGHT_MAX	255

/* Font Sizes */
#define FONT_9X16 0 /**<Small font */
#define FONT_18X16 1 /**<Double width small font */
#define FONT_9X32 2 /**<Double height small font */
#define FONT_18X32 3 /**<Double width and double height small font */
#define FONT_12X24 4 /**<Normal font */
#define FONT_24X24 5 /**<Double width normal font */
#define FONT_12X48 6 /**<Double height normal font */
#define FONT_24X48 7 /**<Double width and double height normal font */

#define FONT_DEFAULT FONT_12X24

/* Printer Model */

/** 
* Enumeration of the supported printers.
*/
enum PRINTERS
{
	PRINTER_PP50=0, /**< Datecs PP-50 mobile printer. */
	PRINTER_PP55, /**< Datecs PP-55 mobile printer. */
	PRINTER_PP60, /**< Datecs PP-60 mobile printer. */
	PRINTER_CMP10, /**< Citizen CMP-10 mobile printer. */
	PRINTER_DPP250, /**< Datecs DPP-250 mobile printer. */
	PRINTER_DPP350, /**< Datecs DPP-350 mobile printer. */
	PRINTER_EP50,  /**< Datecs EP-50 ESC/POS printer. */
	PRINTER_EP55,  /**< Datecs EP-55 ESC/POS printer. */
	PRINTER_EP60, /**< Datecs EP-60 ESC/POS printer. */
	PRINTER_EP300, /**< Datecs EP-300 ESC/POS printer. */
	PRINTER_EP1000, /**< Datecs EP-1000 ESC/POS printer. */
	PRINTER_EP2000, /**< Datecs EP-2000 ESC/POS printer. */
	PRINTER_DPP450, /**< Datecs DPP-450 ESC/POS printer. */
};

#define MCU_CARD 1

/*******************************************************************************
* MIFARE STATUS CODES
*******************************************************************************/ 
#define MF_ERR_OK -10000
#define MF_ERR_TIMEOUT MF_ERR_OK-1
#define MF_ERR_COLLISION MF_ERR_OK-2
#define MF_ERR_PARITY MF_ERR_OK-3
#define MF_ERR_FRAMING MF_ERR_OK-4
#define MF_ERR_CRC MF_ERR_OK-5
#define MF_ERR_FIFO_OVERFLOW MF_ERR_OK-6
#define MF_ERR_EEPROM MF_ERR_OK-7
#define MF_ERR_INVALID_KEY MF_ERR_OK-8
#define MF_ERR_UNKNOWN MF_ERR_OK-9
#define MF_ERR_AUTH MF_ERR_OK-10
#define MF_ERR_CODE MF_ERR_OK-11
#define MF_ERR_BITCOUNT MF_ERR_OK-12
#define MF_ERR_NOT_AUTH MF_ERR_OK-13
#define MF_ERR_VALUE MF_ERR_OK-14

#define MF_REQUEST_IDLE_CARDS 0
#define MF_REQUEST_ALL_CARDS 1

/** @defgroup general General */
/** @{ */

/**
  Defined a financial card structure
*/
#pragma pack(4)
typedef struct FINANCIAL_CARD
{
	char account_number[20];
	char cardholder_name[27];
	int  expiration_year;
	int  expiration_month;
	char service_code[4];
	char discretionary_data[30];
	char reserved[7];
}FINANCIAL_CARD;

#pragma pack()


extern "C" {
/**
 * Opens the serial port and performs initalization of the printer
 *
 * @param port com port number
 * @param speed com port speed
 * @param  hardware if true, turns on RTS/CTS handshaking, false goes back to XON/OFF
 *
 * @return	ERR_OK - printer is ready. <br>
 *			ERR_TIMEOUT - communication error.
 */
int __stdcall dpsdk_OpenPrinter(int port, int speed, int hardware);


/**
  Closes the serial port
  
  @return ERR_OK - printer is ready<br>
  ERR_TIMEOUT - communication error
 */
int __stdcall dpsdk_ClosePrinter();

/**
  Returns the value of the printer sensors
 
  @return Value >=0 gives sensor bits
  else returns ERR_TIMEOUT - communication error
  */
int __stdcall dpsdk_GetPrinterSensors();

/**
  Prints text with specified font/styles.
  This function can act as both simple plain text printing and quite complex printing using internal tags to format the text. \
  The function uses the currently font size and style (or default ones) as well as the aligning, however it allows modifications of them inside the text. Any modification of the settings using the tags will be reverted when function completes execution. For example if you have default font selected before using printText and set bold font inside, it will be reverted to plain when function completes. \
  The tags are control commands used to modify the text printing parameters. They are surrounded by {} brackets. A list of all control tags follows:
  <ul>
  <li> {==} - reverts all settings to their defaults. It includes font size, style, density, aligning
  <li> {=Fx} - selects font size. x ranges from 0 to 7 as follows:
    <ol>
	<li> 0: FONT_9X16
    <li> 1: FONT_18X16
    <li> 2: FONT_9X32
    <li> 3: FONT_18X32
    <li> 4: FONT_12X24
    <li> 5: FONT_24X24
    <li> 6: FONT_12X48
    <li> 7: FONT_24X48
	</ol>
  <li> {=L} - left text aligning
  <li> {=C} - center text aligning
  <li> {=R} - right text aligning
  <li> {=J} - justified text
  <li> {=Rx} - text rotation as follows:
    <ol>
    <li> 0: not rotated
    <li> 1: rotated 90 degrees
    <li> 2: rotated 180 degrees
	</ol>
  <li> {=Ix} - sets density level as follows:
    <ol>
    <li> 0: density 70%
    <li> 1: density 80%
    <li> 2: density 90%
    <li> 3: density 100%
    <li> 4: density 120%
    <li> 5: density 150%
  <li> {=Dxxx} - sets density level in percent (50-200)
  <li> {+/-B} - sets or unsets bold font style
  <li> {+/-I} - sets or unsets italic font style
  <li> {+/-U} - sets or unsets underline font style
  <li> {+/-V} - sets or unsets inverse font style
  <li> {+/-W} - sets or unsets text word-wrapping
  </ul>
  An example of using tags "{=C}Plain centered text\n{=L}Left centered\n{+B}...bold...{-B}{+I}or ITALIC"

  @param printtext pointer to string containing text to be printed
  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_PrintText(const char *printtext);

/**
  Prints formatted text with specified font/styles.
  The function uses standart format specifiers. 
  For setting font/styles see <code>dpsdk_PrintText(const char *printtext)</code>
  
  @param printtext pointer to string containing text to be printed
  @param ... format arguments

  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_PrintTextf(const char *format, ...);


/**
  Prints image file using specified aligning

  @param file - pointer to multibyte string containing the file path and file name of the image file. Supported formats are BMP, JPG, GIF, TIFF, PNG
  @param align - bitmap aligning, one of:
    <ul>
	<li> ALIGN_LEFT(0) - left aligning
    <li> ALIGN_CENTER(1) - center aligning
    <li> ALIGN_RIGHT(2) - right aligning
	</ul>
  @return ERR_OK - operation completed
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_PrintImage(const char *file, int align);


/**
  Prints image file using specified aligning

  @param file - pointer to wide char string containing the file path and file name of the image file. Supported formats are BMP, JPG, GIF, TIFF, PNG
  @param align - bitmap aligning, one of:
    <ul>
	<li> ALIGN_LEFT(0) - left aligning
    <li> ALIGN_CENTER(1) - center aligning
    <li> ALIGN_RIGHT(2) - right aligning
	</ul>
  @return ERR_OK - operation completed
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_PrintImage16(wchar_t *file, int align);

/**
  Prints the logo, stored into the printer (if any)
  @return  ERR_OK - operation completed<br>
  ERR_TIMEOUT - communication error
  */
int __stdcall dpsdk_PrintLogo();

/**
  Sets (or clears) the printer's logo. The logo is persistent.

  @param file pointer to multibyte string containing the file path and file name of the image file. Pass NULL to clear the logo. <br>
	Supported formats are BMP, JPG, GIF, TIFF, PNG. Every printer have limitations of the supported logo sizes, 
you can get the sizes with @PrnGetPrinterInfo function
  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - communication error<br>
  ERR_INVALID_PARAMS - either logo file is invalid, cannot be loaded or the logo exceeds the maximum size
  */
int __stdcall dpsdk_SetLogo(const char *file);


/**
  Sets (or clears) the printer's logo. The logo is persistent.

  @param file pointer to wide char string containing the file path and file name of the image file. Pass NULL to clear the logo. <br>
	Supported formats are BMP, JPG, GIF, TIFF, PNG. Every printer have limitations of the supported logo sizes, 
you can get the sizes with @PrnGetPrinterInfo function
  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - communication error<br>
  ERR_INVALID_PARAMS - either logo file is invalid, cannot be loaded or the logo exceeds the maximum size
*/
int __stdcall dpsdk_SetLogo16(wchar_t *file);

/**
  Changes printing density level

  @param density - density level in percents (50%-200%)

  @return ERR_OK - operation completed
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_SetDensity(int density);

/**
  Uses the printer cutter if the current printer supports it, or it feeds the paper otherwise

  @return  ERR_OK - operation completed<br>
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_CutPaper();

/**
  Feeds the paper X lines

  @param lines lines to feed the paper in pixels (1/203 of the inch)

  @return ERR_OK - operation completed
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_FeedPaper(int lines);

/**
  Sets the various barcode parameters

  @param align barcode aligning, one of:
  <ul>
  <li> ALIGN_LEFT(0) - left aligning
  <li> ALIGN_CENTER(1) - center aligning
  <li> ALIGN_RIGHT(2) - right aligning
  </ul>
  @param scale barcode scaling, one of:
  <ul>
  <li> BAR_SCALE_2(2) - 2 pixel lines
  <li> BAR_SCALE_3(3) - 3 pixel lines
  <li> BAR_SCALE_4(4) - 4 pixel lines
  </ul>
  @param textpos the position of the HRI code, one of:
  <ul>
  <li> BAR_TEXT_NONE(0) - no HRI code is being drawn
  <li> BAR_TEXT_ABOVE(1) - HRI code drawn above the barcode
  <li> BAR_TEXT_BELOW(2) - HRI code drawn below the barcode
  <li> BAR_TEXT_BOTH(3) - HRI code drawn both above and below the barcode
  </ul>
  @param height barcode height, between 1 and 255, default is 162

  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_SetBarcodePrintSettings(int align, int scale, int textpos, int height);

/**
  Prints a barcode with the parameters specified with @PrnBarcodeSettings

  @param type barcode type, one of:
  <ul>
  <li> BAR_UPCA(0) - UPC-A
  <li> BAR_UPCE(1) - UPC-E
  <li> BAR_EAN13(2) - EAN-13
  <li> BAR_EAN8(3) - EAN-8
  <li> BAR_CODE39(4) - Code 39
  <li> BAR_ITF(5) - ITF
  <li> BAR_CODABAR(6) - Codabar
  <li> BAR_CODE93(7) - Code 93
  <li> BAR_CODE128(8) - Code 128
  <li> BAR_PDF417(9) - PDF-417
  </ul>
  @param data pointer to barcode data
  @param len the size of the barcode data

  @return ERR_OK - operation completed<br>
  ERR_INVALID_PARAMS - some of the input parameters contain wrong value<br>
  ERR_UNSUPPORTED - barcode type is not supported<br>
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_PrintBarcode(int type, const char *data, int len);

/**
  Prints PDF-417 barcode

  @param type PDF-417 type, one of:
  <ul>
  <li> 0 - Standard
  <li> 1 - Truncated
  </ul>
  @param encoding encoding type, one of:
  <ul>
  <li> 0 - Automatic
  <li> 1 - Binary
  </ul>
  @param eccl Error correction control level. Possible values 0 to 9. ECCL=9 automatically selects correction level dependent on data length.
  @param size barcode size, one of:
  <ul>
  <li> 0 - Width=2, Height=4
  <li> 1 - Width=2, Height=9
  <li> 2 - Width=2, Height=15
  <li> 3 - Width=2, Height=20
  <li> 4 - Width=7, Height=4
  <li> 5 - Width=7, Height=9
  <li> 6 - Width=7, Height=15
  <li> 7 - Width=7, Height=20
  <li> 8 - Width=12, Height=4
  <li> 9 - Width=12, Height=9
  <li> 10 - Width=12, Height=15
  <li> 11 - Width=12, Height=20
  <li> 12 - Width=20, Height=4
  <li> 13 - Width=20, Height=9
  <li> 14 - Width=20, Height=15
  <li> 15 - Width=20, Height=20
  </ul>
  @param data pointer to barcode data
  @param datalen the size of the barcode data

  @return ERR_OK - operation completed<br>
  ERR_INVALID_PARAMS - some of the input parameters contain wrong value<br>
  ERR_UNSUPPORTED - barcode type is not supported<br>
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_PrintBarcodePDF417(int type, int encoding, int eccl, int size, const unsigned char *data, int datalen);

/**
  Prints QR CODE barcode

  @param size barcode symbol size. Possible values: 1, 4, 6, 8, 10, 12, 14
  @param eccl Error correction control level, one of:
  <ul>
  <li> 1 - 7%
  <li> 2 - 15%
  <li> 3 - 25%
  <li> 4 - 30%
  <\ul>
  @param data pointer to barcode data
  @param datalen the size of the barcode data

  @return ERR_OK - operation completed<br>
  ERR_INVALID_PARAMS - some of the input parameters contain wrong value<br>
  ERR_UNSUPPORTED - barcode type is not supported<br>
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_PrintBarcodeQRCode(int size, int eccl, const unsigned char *data, int datalen);

/**
  Changes active code page if possible<br>
  Some printers require manually enabling this with hardware switch (look for ESC t in the printer's manual)

  @param codepage code page identifier<br>
  OEM code pages:
  <ul>
  <li> 437 - IBM PC
  <li> 737 - Greek
  <li> 775 - Estonian, Lithuanian and Latvian
  <li> 850 - "Multilingual (Latin-1)" (Western European languages)
  <li> 852 - "Slavic (Latin-2)" (Central and Eastern European languages)
  <li> 856 - Cyrillic
  <li> 857 - Turkish
  <li> 860 - Portuguese
  <li> 862 - Hebrew
  <li> 866 - Cyrillic
  </ul>
  Windows ANSI code pages
  <ul>
  <li> 1250 - Central and East European Latin
  <li> 1251 - Cyrillic
  <li> 1252 - West European Latin
  <li> 1253 - Greek
  <li> 1254 - Turkish
  <li> 1255 - Hebrew
  <li> 1257 - Baltic
  </ul>

  @return ERR_OK - operation completed
  ERR_TIMEOUT - communication error
  ERR_UNSUPPORTED - the printer does not support the codepage selected
*/
int __stdcall dpsdk_SetCodepage(int codepage);

/**
  Returns current printer's paper width in pixels (1/203 of inch)

  @return Paper width in pixels.
*/
int __stdcall dpsdk_GetPaperWidth();

/**
  Draws a line with a specified font and char. The line occupies the whole paper width so it is useful for delimiters

  @param delimchar the symbol to be drawn, usually '-'
  @parma font font style, one of the:
  <ul>
  <li> FONT_9X16(0) - Font 9x16
  <li> FONT_18X16(1) - Font 18x16
  <li> FONT_9X32(2) - Font 9x32
  <li> FONT_18X32(3) - Font 18x32
  <li> FONT_12X24(4) - Font 12x24 (default)
  <li> FONT_24X24(5) - Font 24x24
  <li> FONT_12X48(6) - Font 12x48
  <li> FONT_24X48(7) - Font 24x48
  </ul>

  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - communication error
*/
int __stdcall dpsdk_PrintDelimiter(char delimchar, int font);

/**
  Calculates the complete character width (plus intercharacter space) of a single character at given font

  @param font font style, one of the:
  <ul>
  <li> FONT_9X16(0) - Font 9x16
  <li> FONT_18X16(1) - Font 18x16
  <li> FONT_9X32(2) - Font 9x32
  <li> FONT_18X32(3) - Font 18x32
  <li> FONT_12X24(4) - Font 12x24 (default)
  <li> FONT_24X24(5) - Font 24x24
  <li> FONT_12X48(6) - Font 12x48
  <li> FONT_24X48(7) - Font 24x48
  </ul>

  @return Character width in pixels
*/
int __stdcall dpsdk_GetPrinterCharWidth(int font);

/**
  Calculates the complete character height of a single character at given font

  @param font font style, one of the:
  <ul>
  <li> FONT_9X16(0) - Font 9x16
  <li> FONT_18X16(1) - Font 18x16
  <li> FONT_9X32(2) - Font 9x32
  <li> FONT_18X32(3) - Font 18x32
  <li> FONT_12X24(4) - Font 12x24 (default)
  <li> FONT_24X24(5) - Font 24x24
  <li> FONT_12X48(6) - Font 12x48
  <li> FONT_24X48(7) - Font 24x48
  </ul>

  @return Character height in pixels
*/
int __stdcall dpsdk_GetPrinterCharHeight(int font);

/**
  Calculates the complete text width (plus intercharacter space) at given font

  @param text null-terminated string to calculate the width of
  @param font font style, one of the:
  <ul>
  <li> FONT_9X16(0) - Font 9x16
  <li> FONT_18X16(1) - Font 18x16
  <li> FONT_9X32(2) - Font 9x32
  <li> FONT_18X32(3) - Font 18x32
  <li> FONT_12X24(4) - Font 12x24 (default)
  <li> FONT_24X24(5) - Font 24x24
  <li> FONT_12X48(6) - Font 12x48
  <li> FONT_24X48(7) - Font 24x48
  </ul>

  @return Text width in pixels
*/
int __stdcall dpsdk_GetTextWidth(const char *text, int font);

/**
  Returns the currently connected printer model

  @return Printer model:
  <ul>
  <li> PRINTER_PP50(0) - PP-50
  <li> PRINTER_PP55(1) - PP-55
  <li> PRINTER_PP60(2) - PP-60
  <li> PRINTER_CMP10(3) - CMP-10
  <li> PRINTER_DPP250(4) - DPP-250
  <li> PRINTER_DPP350(5) - DPP-350
  <li> PRINTER_EP50(6) - EP-50
  <li> PRINTER_EP55(7) - EP-55
  <li> PRINTER_EP60(8) - EP-60
  <li> PRINTER_EP300(9) - EP-300
  <li> PRINTER_EP1000(10) - EP-1000
  <li> PRINTER_EP2000(11) - EP-2000
  </ul>
*/
int __stdcall dpsdk_GetPrinterInfo(void *information);

/**
  Returns the currently connected printer information.
int __stdcall dpsdk_GetPrinterModel();

/** @} */

/** @defgroup msr Magnetic Stripe Reader */
/** @{*/


/**
  Reads magnetic stripe card
  
  @param track1 buffer for holding returned card data, minimum of 80 bytes. Pass NULL if you don't want track 1 contents.
  @param track2 buffer for holding returned card data, minimum of 41 bytes. Pass NULL if you don't want track 2 contents.
  @param track3 buffer for holding returned card data, minimum of 108  bytes. Pass NULL if you don't want track 3 contents.
  @param timeout timeout in miliseconds to read the card data. The actuall scan time may differ, 
but will be as close as possible to this value

  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - timeout reading card<br>
  ERR_UNSUPPORTED - the printer connected does not support reading magnetic cards
*/
int __stdcall dpsdk_msr_ReadCard(char *track1, char *track2, char *track3, int timeout);

/**
  Extracts finanial data (such as name, number, expiration date) from financial magnetic card

  @param track1 track 1 data returned from @PrnMSReadCard or NULL
  @param track2 track 2 data returned from @PrnMSReadCard or NULL
  @param data pointer to FINANCIAL_CARD structure. Every field that can be decoded will be filled.

  @return ERR_OK - operation completed<br>
  ERR_UNSUPPORTED - the card data passed is not from a valid financial card
*/
int __stdcall dpsdk_msr_ProcessFinancialCard(char *track1, char *track2, FINANCIAL_CARD *data);

/** @}*/

/** @defgroup scr Smart Card Reader */
/** @{ */

/**
  Initializes and powers up smartcard reader module

  @return ERR_OK - operation completed<br>
  ERR_UNSUPPORTED - no smartcard reader module present
*/
int __stdcall dpsdk_scr_SetPowerOn();

/**
  Powers down smartcard reader module

  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - error communicating with the reader
*/
int __stdcall dpsdk_scr_SetPowerOff();

/**
  Resets the smartcard and returns the ATR. Call this function before sending any APDU commands

  @param protocol card communication protocol, one of:
  <ul>
  <li> 0 - T0
  <li> 1 - T1
  <\ul>
  @param atr data buffer where the ATR is returned
  @param atrSize pass the size of atr buffer, upon exit it will contain the count of the bytes written in the atr buffer. 
If atr buffer is not big enough, then this function will return ERR_BUFFER_TOO_SMALL and <code>atr_length</code> will contain the bytes needed

  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - error communicating with the reader<br>
  ERR_BUFFER_TOO_SMALL - the size of atr buffer is not enough
*/
int __stdcall dpsdk_scr_Reset(int *protocol, unsigned char *atr, int *atrSize);

/**
  Sends APDU write request

  @param cla the CLA parameter. Consult smartcard documentation.
  @param ins the INS parameter. Consult smartcard documentation.
  @param p1 the P1 parameter. Consult smartcard documentation.
  @param p2 the P2 parameter. Consult smartcard documentation.
  @param inBuf pointer to a byte array to write to the card or NULL if you don't want to send any data with the command
  @param inBufSize the length of the data, pointed by write_buffer

  @return ERR_OK - operation completed<br>
  ERR_TIMEOUT - error communicating with the reader
*/
int __stdcall dpsdk_scr_Write(int cla, int ins, int p1, int p2, const unsigned char* inBuf, int inBufSize);

/**
  Sends APDU read request

  @param cla the CLA parameter. Consult smartcard documentation.
  @param ins the INS parameter. Consult smartcard documentation.
  @param p1 the P1 parameter. Consult smartcard documentation.
  @param p2 the P2 parameter. Consult smartcard documentation.
  @param outBuf pointer to a byte array to write to store the returned card information
  @param outBufMaxSize the length of the data (0-255 bytes) you want to read
  
  @return Result value greater or equal to zero represents the bytes of data returned from the card, negative value means error.<br>
  ERR_OK - operation completed<br>
  ERR_TIMEOUT - error communicating with the reader
*/
int __stdcall dpsdk_scr_Read(int cla, int ins, int p1, int p2, int lc, unsigned char* outBuf, int outBufMaxSize);

/**
  Sends combined APDU read/write request

  @param cla the CLA parameter. Consult smartcard documentation.
  @param ins the INS parameter. Consult smartcard documentation.
  @param p1 the P1 parameter. Consult smartcard documentation.
  @param p2 the P2 parameter. Consult smartcard documentation.
  @param inBuf pointer to a byte array to write to the card or NULL if you don't want to send any data with the command
  @param inBufSize the length of the data, pointed by write_buffer
  @param outBuf pointer to a byte array to write to store the returned card information
  @param outBufMaxSize the length of the data (0-255 bytes) you want to read

  @return Result value greater or equal to zero represents the bytes of data returned from the card, negative value means error.
  ERR_OK - operation completed
  ERR_TIMEOUT - error communicating with the reader
*/
int __stdcall dpsdk_scr_Transmit(int cla, int ins, int p1, int p2, const unsigned char* inBuf, int inBufSize, unsigned char* outBuf, int outBufMaxSize);

/**
  Returns if smart card is present into reader.

  @param yes a variable that receives if card is present.

  @return On success returns DATECS_ENONE otherwise returns an error code.
*/
int __stdcall dpsdk_scr_IsCardPresent(int* yes);

/**
  Selects the smart card reader slot.

  @param slot the smart card reader slot number, starting from 0.

  @return On success returns DATECS_ENONE otherwise returns an error code.
*/
int __stdcall dpsdk_scr_Select(int slot);

/**
  Returns the value of the status bytes SW1 and SW2 as a value between 0 and 255.

  @param sw1 the value of the status bytes SW1 as a value between 0 and 255.
  @param sw2 the value of the status bytes SW2 as a value between 0 and 255.

  @return On success returns DATECS_ENONE otherwise returns an error code.
*/
int __stdcall dpsdk_scr_GetStatus(int* sw1, int* sw2);
/** @} */

/** /defgrpoup mfr MiFare Reader */
/** @{ */
/**
  Returns mifare reader identification string. You have to call @pdsdk_mfr_SetPowerOn first

  @param ident - returns mifare reader identification string (max 32 characters)

  @return ERR_OK - operation completed<br>
  MF_STAT_TIMEOUT - mifare reader failed to respond
*/
int __stdcall dpsdk_mfr_GetIdentification(char *ident);

/**
  Opens Mifare Reader channel, call this function before any other Mifare Reader one.<br>
  The next mifare function must be <code>int dpsdk_mfr_Config()</code> to power on and configure the mifare reader.

  @return ERR_OK - operation completed<br>
  MF_STAT_TIMEOUT - mifare reader failed to respond
*/
int __stdcall dpsdk_mfr_SetPowerOn();

/**
  Initializes and powers on the mifare reader module. Use @dpsdk_mfr_SetPowerOn before calling this function.
  Call this function before any other mifare functions, except @dpsdk_mfr_SetPowerOn.

  @return ERR_OK - operation completed<br>
  MF_STAT_TIMEOUT - mifare reader failed to respond
*/
int __stdcall dpsdk_mfr_Config();

/**
  Powers down mifare reader module. Call this function after you are done with the mifare reader.

  @return ERR_OK - operation completed<br>
  MF_STAT_TIMEOUT - mifare reader failed to respond
*/
int __stdcall dpsdk_mfr_SetPowerOff();

/**
  Returns the serial number of one of requested Mifare cards.

  @param serialNumber the variable that receives card serial number.

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_AntiCollision(unsigned int *serialNumber);

/**
  Scans for mifare cards in the area

  @param all card communication protocol, one of:
  <ul>
  <li> MF_REQUEST_IDLE_CARDS - Return only cards that are not halted
  <li> MF_REQUEST_ALL_CARDS - Return all cards
  <\ul>
  @param atqa the ATQA data.

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_Request(int all, int *atqa);

/**
  Selects a card to operate with

  @param serialNumber card serial number, returned from <code>dpsdk_mfr_Request</code>
  @param sack the variable that receives SAK (Select Acknowledge) value.

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_Select(unsigned int serialNumber, int *sack);

/**
  Puts the selected with @MFSelectCard card into halted state, such card will not show from @pdsdk_mfr_Request with type set to MF_REQUEST_IDLE_CARDS

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_Halt();

/**
  Authenticate with mifare card

  @param serialNumber the serial number of Mifare card.
  @param block the block number of Mifare card requested for authentication.
  @param type key type, either 'A' or 'B'
  @patam key 6 bytes key

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_Authenticate(unsigned int serialNumber, int block, int keyType, const unsigned char key[]);

/**
  Authenticates Mifare card for operation with key stored into Mifare reader EEPROM memory.

  @param serialNumber the serial number of Mifare card.
  @param block the block number of Mifare card requested for authentication.
  @param type key type, either 'A' or 'B'
  @param selector the number of key selector.

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_AuthenticateEX(unsigned int serialNumber, int block, int keyType, int selector);

/**
  Reads a 16 byte block of data

  @param block the block number.
  @param data buffer to return the 16 bytes of data if successful

  @return ERR_OK operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_Read(int block, unsigned char data[]);

/**
  Writes a 16 byte block of data

  @param block address to write to
  @param data 16 bytes block of data  
  
  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_WriteData(int block, const unsigned char data[]);

/**
  Writes int data

  @param block address to write to
  @param data the data

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_WriteValue(int block, int data);

/**
  Performs increment operations on value cards

  @param source source block number
  @param destination destination block number
  @param value the value to increment with

  @Return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_Increment(int source, int destination, int value);

/**
  Performs decrement operations on value cards

  @param source source block number
  @param destination destination block number
  @param value the value to decrement with

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_Decrement(int source, int destination, int value);

/**
  Performs restore operations on value cards

  @param source source block number
  @param destination destination block number
  @param value the value to restore with

  @param ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_Restore(int source, int destination, int value);

/**
  Returns mifare reader's serial number

  @param serialNumber where the 4 byte serial number will be returned

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_GetSerialNumber(unsigned int *serialNumber);

/**
  Stores key into Mifare reader EEPROM.

  @param selector the key selector number.
  @param key the byte buffer contains 6 bytes authentication key.

  @return ERR_OK - operation completed, or one of the MF_STAT_* constants upon error
*/
int __stdcall dpsdk_mfr_LoadKey(int selector, const unsigned char key[]);

/** @} */

/** @defgroup bcr Barcode Reader */
/** @{ */

/**
  Helper function to return string name of barcode type

  @param type barcode type returned from @BarcodeScan
  @param text string, where the barcode type is supported. Make sure it is at least 32 bytes long
*/
void __stdcall dpsdk_bcr_Type2Text(int type, char *text);

/**
  Scans barcode using the built-in barcode scanning engine
  
  @param type upon success, barcode type is returned here, one of the BARCODE_* constants
  @param barcode string, where the barcode data is returned.
  @param maxlen the size of the buffere pointed by barcode parameter

  @return ERR_OK - barcode successfully scanned<br>
  ERR_UNSUPPORTED - barcode reader not supported, or barcode engine error<br>
  ERR_TIMEOUT - timeout scanning barcode
*/
int __stdcall dpsdk_bcr_ScanBarcode(int *type, char *barcode, int maxlen);
/** @} */


/**
@ingroup general
  Sets the ASCII code that will be used to output euro sign.

  @param code ASCII code for euro sign (0-255). Passing 0 disables the euro sign. Default is disabled.

  @return ERR_OK - operation completed<br>
  ERR_UNSUPPORTED - this printer does not support defining custom euro code<br>
  ERR_TIMEOUT - timeout sending the data<br>
  ERR_INVALID_PARAMS - code parameter have wrong value<br>
*/
int __stdcall dpsdk_SetEuroCode(int code);

/**
@ingroup general
  Writes the buffer directly to the printer

  @param buffer the data to be send
  @param bufferSize size of the buffer

  @return ERR_OK - data was successfully sent
  ERR_UNSUPPORTED - channel not supported on the current printer/configuration
*/
int __stdcall dpsdk_WriteBuffer(const void* buffer, int bufferSize);

/**
@ingroup general
  Gets a specified setting from the printer

  @param setting The setting to be retrieved. Possible values are:<br>
  Identification
  <ul>
  <li> "Identification:Name"
  <li> "Identification:FirmwareVersion"
  <li> "Identification:BatteryLevel"
  <li> "Identification:BatteryLevelPercent"
  <li> "Identification:HeadTemperature"
  <li> "Identification:Capabilities"<br>
		Value format: <Capability 0>:<Supported flag>,<Capability 1>:<Supported flag>, ... ,<Capability 13>:<Supported flag>
		Supported flag
		<ol>
		<li> "0" - the capability is NOT supported
		<li> "1" - the capability is suported
		</ol>
		Capability value(ID)
		<ol>
		<li>  "0" - Has Magnetic Stripe Reader 
		<li>  "1" - Can read Magnetic Stripe third track
		<li>  "2" - Barcode Reader
		<li>  "3" - Can update firmware
		<li>  "4" - Has IRDA
		<li>  "5" - Has Bluetooth
		<li>  "6" - Has USB A
		<li>  "7" - Has USB B
		<li>  "8" - Katakana symbols support
		<li>  "9" - Korean symbols support
		<li> "10" - Head temperature returned in Farenheit
		<li> "11" - Black mark paper support
		<li> "12" - Has WiFi
		<li> "13" - Has LAN
		</ol>
  <li> "Identification:UniversalReader"
  <li> "Identification:SerialNumber"
  <li> "Identification:AvailablePrintSpeeds"
  </ul>
  Sound
  <ul>
  <li> "Sound:PowerOn"
  <li> "Sound:MSR"
  <li> "Sound:Barcode"
  <li> "Sound:Bell"
  </ul>
  Print
  <ul>
  <li> "Print:Density"
  <li> "Print:Font"
  <li> "Print:LineSpace"
  <li> "Print:Speed"
  <li> "Print:Country"
  <li> "Print:CodePage"
  <li> "Print:EuroPosition"
  <li> "Print:CRasLF"
  <li> "Print:DisableLF"
  <li> "Print:DisableLFAfterCR"
  </ul>
  USB
  <ul>
  <li> "USB:Class"
  <li> "USB:VendorID"
  <li> "USB:ProductIDPrinter"
  <li> "USB:ProductIDSerial"
  <li> "USB:Manufacturer"
  <li> "USB:Model"
  <li> "USB:Device"
  </ul>
  USB A
  <ul>
  <li> "USBA:Enabled"
  <li> "USBA:SlaveMode"
  <li> "USBA:RawMode"
  <li> "USBA:RemoteWakeup"
  </ul>
  USB B
  <ul>
  <li> "USBB:Enabled"
  <li> "USBB:SlaveMode"
  <li> "USBB:RawMode"
  <li> "USBB:RemoteWakeup"
  </ul>
  General
  <ul>
  <li> "General:RTC"
  <li> "General:Timeout"
  <li> "General:USBTimeout"
  <li> "General:PortSpeed"
  <li> "General:DisableCoverOpen"
  <li> "General:ProtocolMode"
  <li> "General:XonXoff"
  <li> "General:DisableCutter"
  <li> "General:DefaultDisplay"
  <li> "General:SmallRoll"
  <li> "General:DisableESCt"
  <li> "General:ExtendedCommands"
  <li> "General:BlackMarkMode"
  </ul>
  Bluetooth
  <ul>
  <li> "Bluetooth:Enabled"
  <li> "Bluetooth:Discoverable"
  <li> "Bluetooth:SavePairingInformation"
  <li> "Bluetooth:PairingKey"
  <li> "Bluetooth:DeviceName"
  </ul>
  WiFi
  <ul>
  <li> "WiFi:Enabled"
  <li> "WiFi:IPAddress"
  <li> "WiFi:SubnetMask"
  <li> "WiFi:DefaultGateway"
  <li> "WiFi:EnableDHCP"
  <li> "WiFi:LocalPort"  
  <li> "WiFi:NetworkMode"
  <li> "WiFi:IBSSChannel"
  <li> "WiFi:RegionalDomain"
  <li> "WiFi:NetworkSSID"
  <li> "WiFi:PowerManagement"
  <li> "WiFi:Password"
  </ul>
  LAN
  <ul>
  <li> "LAN:Enabled"
  <li> "LAN:EnableDHCP"
  <li> "LAN:LocalPort"
  <li> "LAN:IPAddress"
  <li> "LAN:SubnetMask"
  <li> "LAN:DefaultGateway"
  <li> "LAN:MACAddress"
  </ul>
  @param value Null terminated string containing the returned value or empty string if the setting is not supported.
  The value depands on the setting requested.

  @param maxSize Maximum size of the <code>value</code> parameter
  @return ERR_OK - operation completed<br>
  ERR_UNSUPPORTED - the requested setting is not supported by the connected device.
		
*/
int __stdcall dpsdk_GetPrinterSetting(const char* setting, char* value, int maxSize);

/**
@ingroup general
  Sets a specified setting in the library's buffer. All settings stored in the libraty's buffer are set in the printer using <code>dpsdk_CommitPrinterSettings()</code>

  @param setting The setting to be set. Possible values are:<br>
  Identification
  <ul>
  <li> "Identification:Name"
  <li> "Identification:FirmwareVersion"
  <li> "Identification:BatteryLevel"
  <li> "Identification:BatteryLevelPercent"
  <li> "Identification:HeadTemperature"
  <li> "Identification:Capabilities"<br>
		Value format: <Capability 0>:<Supported flag>,<Capability 1>:<Supported flag>, ... ,<Capability 13>:<Supported flag>
		Supported flag
		<ol>
		<li> "0" - the capability is NOT supported
		<li> "1" - the capability is suported
		</ol>
		Capability value(ID)
		<ol>
		<li>  "0" - Has Magnetic Stripe Reader 
		<li>  "1" - Can read Magnetic Stripe third track
		<li>  "2" - Barcode Reader
		<li>  "3" - Can update firmware
		<li>  "4" - Has IRDA
		<li>  "5" - Has Bluetooth
		<li>  "6" - Has USB A
		<li>  "7" - Has USB B
		<li>  "8" - Katakana symbols support
		<li>  "9" - Korean symbols support
		<li> "10" - Head temperature returned in Farenheit
		<li> "11" - Black mark paper support
		<li> "12" - Has WiFi
		<li> "13" - Has LAN
		</ol>
  <li> "Identification:UniversalReader"
  <li> "Identification:SerialNumber"
  <li> "Identification:AvailablePrintSpeeds"
  </ul>
  Sound
  <ul>
  <li> "Sound:PowerOn"
  <li> "Sound:MSR"
  <li> "Sound:Barcode"
  <li> "Sound:Bell"
  </ul>
  Print
  <ul>
  <li> "Print:Density"
  <li> "Print:Font"
  <li> "Print:LineSpace"
  <li> "Print:Speed"
  <li> "Print:Country"
  <li> "Print:CodePage"
  <li> "Print:EuroPosition"
  <li> "Print:CRasLF"
  <li> "Print:DisableLF"
  <li> "Print:DisableLFAfterCR"
  </ul>
  USB
  <ul>
  <li> "USB:Class"
  <li> "USB:VendorID"
  <li> "USB:ProductIDPrinter"
  <li> "USB:ProductIDSerial"
  <li> "USB:Manufacturer"
  <li> "USB:Model"
  <li> "USB:Device"
  </ul>
  USB A
  <ul>
  <li> "USBA:Enabled"
  <li> "USBA:SlaveMode"
  <li> "USBA:RawMode"
  <li> "USBA:RemoteWakeup"
  </ul>
  USB B
  <ul>
  <li> "USBB:Enabled"
  <li> "USBB:SlaveMode"
  <li> "USBB:RawMode"
  <li> "USBB:RemoteWakeup"
  </ul>
  General
  <ul>
  <li> "General:RTC"
  <li> "General:Timeout"
  <li> "General:USBTimeout"
  <li> "General:PortSpeed"
  <li> "General:DisableCoverOpen"
  <li> "General:ProtocolMode"
  <li> "General:XonXoff"
  <li> "General:DisableCutter"
  <li> "General:DefaultDisplay"
  <li> "General:SmallRoll"
  <li> "General:DisableESCt"
  <li> "General:ExtendedCommands"
  <li> "General:BlackMarkMode"
  </ul>
  Bluetooth
  <ul>
  <li> "Bluetooth:Enabled"
  <li> "Bluetooth:Discoverable"
  <li> "Bluetooth:SavePairingInformation"
  <li> "Bluetooth:PairingKey"
  <li> "Bluetooth:DeviceName"
  </ul>
  WiFi
  <ul>
  <li> "WiFi:Enabled"
  <li> "WiFi:IPAddress"
  <li> "WiFi:SubnetMask"
  <li> "WiFi:DefaultGateway"
  <li> "WiFi:EnableDHCP"
  <li> "WiFi:LocalPort"  
  <li> "WiFi:NetworkMode"
  <li> "WiFi:IBSSChannel"
  <li> "WiFi:RegionalDomain"
  <li> "WiFi:NetworkSSID"
  <li> "WiFi:PowerManagement"
  <li> "WiFi:Password"
  </ul>
  LAN
  <ul>
  <li> "LAN:Enabled"
  <li> "LAN:EnableDHCP"
  <li> "LAN:LocalPort"
  <li> "LAN:IPAddress"
  <li> "LAN:SubnetMask"
  <li> "LAN:DefaultGateway"
  <li> "LAN:MACAddress"
  </ul>
  @param value Null terminated string containing the returned value or empty string if the setting is not supported.
  The value depands on the setting requested.

  @return ERR_OK - operation completed<br>
  ERR_UNSUPPORTED - the requested setting is not supported by the connected device.
		
*/
int __stdcall dpsdk_SetPrinterSetting(const char* setting, const char* value);

/**
@ingroup general
Commits the stored settings in the printer. The settings can be set using <code>dpsdk_SetPrinterSetting(const char* setting, const char* value)</code>

@return ERR_OK - operation completed<br>
  ERR_TIMEOUT - timeout sending the data<br>
*/
int __stdcall dpsdk_CommitPrinterSettings();

}
