package connectivity;

import com.datecs.fiscalprinter.SDK.FiscalDeviceV2;
import com.datecs.fiscalprinter.SDK.FiscalException;
import com.datecs.fiscalprinter.SDK.model.BGR.*;
import com.datecs.fiscalprinter.SDK.model.DatecsFiscalDevice;
import com.datecs.util.LoggerEx;
import com.datecs.util.Translate;


import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.util.StringTokenizer;
import java.util.logging.Level;

import static com.datecs.fiscalprinter.SDK.FiscalException.FPTR_CC_OTHER;


/**
 * @author DATECS Software Dep.
 */
public class SocketConnector implements PrinterConnection {

    private int timeout = 0;
    private final Socket socket;
    private InputStream in;
    private OutputStream out;

    public static String getModelName() {
        return modelName;
    }

    private String portName = "";
    final private LoggerEx logger = LoggerEx.getLogger(SocketConnector.class);
    private static DatecsFiscalDevice connectedFD = new DatecsFiscalDevice(FPTR_CC_OTHER);
    private static String modelName;

    public static DatecsFiscalDevice getConnectedFD() throws Exception {
        if (connectedFD.isConnectedDeviceV2()) return connectedFD;
        else throw new Exception("There is no supported Device connected");
    }


    public SocketConnector() {
        socket = new Socket();
    }

    void waitConnected(int timeout) throws Exception {
        long startTime = System.currentTimeMillis();
        for (; ; ) {
            Thread.sleep(0, 001);
            long currentTime = System.currentTimeMillis();
            if (socket.isConnected()) {
                break;
            }
            if ((currentTime - startTime) > timeout) {
                noConnectionError();
            }
        }
    }

    @Override
    public void close() throws Exception {
        socket.shutdownInput();
        socket.shutdownOutput();
        socket.close();
    }

    public int readByte() throws Exception {
        int b = doReadByte();
        return b;
    }

    public int doReadByte() throws Exception {
        int result;
        long startTime = System.currentTimeMillis();
        for (; ; ) {
            Thread.sleep(0, 001);
            long currentTime = System.currentTimeMillis();
            if (in.available() > 0) {
                result = in.read();
                if (result >= 0) {
                    return result;
                }
            }
            if ((currentTime - startTime) > timeout) {
                noConnectionError();
            }
        }
    }

    private void noConnectionError() throws Exception {
        throw new IOException(Translate.getString(Translate.NoConnection));
    }

    public byte[] readBytes(int len) throws Exception {
        byte[] data = new byte[len];
        int offset = 0;
        while (len > 0) {
            int count = in.read(data, offset, len);
            if (count == -1) {
                break;
            }
            len -= count;
            offset += count;
        }
        return data;
    }

    /**
     * @throws Exception
     */
    public void connect() throws Exception {

    }

    public void write(byte[] b) throws Exception {

    }

    public void write(int b) throws Exception {

    }

    public void setBaudRate(int baudRate) throws Exception {

    }

    public void setTimeout(int timeout) throws Exception {
        this.timeout = timeout;
    }

    public void setPortName(String portName) throws Exception {
        this.portName = portName;
    }

    public InputStream getInputStream() throws Exception {
        return socket.getInputStream();
    }

    public OutputStream getOutputStream() throws Exception {
        return socket.getOutputStream();
    }

    public Object getSyncObject() throws Exception {
        return socket;
    }

    public boolean isSearchByBaudRateEnabled() {
        return false;
    }

    @Override
    public DatecsFiscalDevice open(String portName) throws Exception {
        try {

            socket.setSoTimeout(this.timeout); //TODO:
            StringTokenizer tokenizer = new StringTokenizer(portName, ":");
            String host = tokenizer.nextToken();
            int port = Integer.parseInt(tokenizer.nextToken());
            while (true) {
                if (socket.isConnected()) break;
                socket.connect(new InetSocketAddress(host, port));
                Thread.sleep(100);
            }
            in = socket.getInputStream();
            out = socket.getOutputStream();
            /**
             *  Datecs JavaSDK types of exceptions:
             *
             *   The FiscalException are throws in Datecs JavaSDK,
             *
             *  FiscalException of Status Bytes they depend on whether a status is marked as critical or not,
             *  the SDK method setIsStatusCritical() - enables SDK users to select Status of device
             *  to be a critical exception for their application or not.
             *  For example, Status of (Byte[0] Bit[3])-"No client display connected"
             *  may be a critical error in a POS application running in a store but
             *  if fiscal printer is an office application and no need of client display, this exception must to turned off.
             *
             *  FiscalException - Command Error
             *  These are all exceptions for which a Fiscal Device command can not be executed,
             *  after the execution of each command, the device returns a response contain error code,
             *  If the error code is different 0-OK, Datecs JavaSDK - converts the code into messages and added it to
             *
             *   Two setters define the content of the message:
             *   setThrowErrorCode() -about the error code.
             *   setThrowErrorMessage()-for the description of the error.
             *
             * Use fiscalDevice.getConnectedPrinterV2().setMsgSeparator(); to setup desired separator char or string,
             * by default separator is "\n\r"
             *
             * All messages are in English.
             */

            //Enables whether a status of the device is a critical exception for user application !
            FiscalDeviceV2.setIsStatusCritical(initCriticalStatus());
            //Autodetect model
            FDModelDetectorV2 datecsBGRmodelV2 = new FDModelDetectorV2(in, out);
            modelName = datecsBGRmodelV2.detectConnectedModel();

            switch (modelName) {
                case "FMP-350X":
                    connectedFD.setConnectedModel(new FMP350X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                    return connectedFD;
                case "FMP-55X":
                    connectedFD.setConnectedModel(new FMP55X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                    return connectedFD;
                case "FP-700X":
                    connectedFD.setConnectedModel(new FP700X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                    return connectedFD;
                case "WP-500X":
                    connectedFD.setConnectedModel(new WP500X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                    datecsBGRmodelV2.getTransportProtocol().setDebugLevel(Level.OFF);
                    return connectedFD;
                case "WP-50X":
                    connectedFD.setConnectedModel(new WP50X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                    datecsBGRmodelV2.getTransportProtocol().setDebugLevel(Level.ALL);
                    return connectedFD;
                case "DP-25X":
                    connectedFD.setConnectedModel(new DP25X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                    return connectedFD;
                case "DP-150X":
                    connectedFD.setConnectedModel(new DP150X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                    return connectedFD;
                default: {
                    throw new Exception("Unsupported model:" + modelName);
                }
            }

        } catch (
                SocketException e) {
            e.printStackTrace();
            throw new Exception("SocketException :" + e.getMessage());
        } catch (IOException e) {
            //   Thread.sleep(1000);
            e.printStackTrace();
            throw new Exception("IOException :" + e.getMessage());
        } catch (
                FiscalException e) {
            e.printStackTrace();
            throw new Exception("FiscalException :" + e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new Exception("Exception :" + e.getMessage());
        }


    }

    @Override
    public String getPortName() throws Exception {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }


}
